#!/bin/bash
DEBUG=y

OK=0
CANCEL=1
HELP=2
#RETURN=254
ESC=255
BACKTITLE="Moxa Connection Manager (MCM)"

CONF_PATH=/etc/moxa/MoxaConnectionManager/
INTERFACE_CONF_PATH=${CONF_PATH}/interfaces/
NEW_PROFILE_PATH=/lib/moxa/MoxaConnectionManager/script/

#FAILOVER_LIST=()
declare -A FAILOVER_LIST

INTERFACE=
INTERFACE_DEVICE_TYPE=
INTERFACE_PROFILE_LIST=
T_PROFILE=
declare -A INTERFACE_LIST
declare -A INTERFACE_AL_KEEPALIVE_LIST

MODEM_SUPPORT_RAT_LIST=()

##### Main Configure ######
LOG_LEVEL=
FAILBACK_ENABLED=
FAILBACK_CHECK_INTERVAL=
FAILOVER_SEAMLESS_ENABLED=
BRIDGE_DHCP_ENABLED=
BRIDGE_IPV4_ADDRESS=
BRIDGE_IPV4_NETMASK=
BRIDGE_IP_RANGE=
BRIDGE_IP_LEASE_TIME=
GLOBAL_DNS_SERVERS=
DHCP_SERVER=

IPADDRESS_REGEX="^(25[0-5]|2[0-4][0-9]|1[0-9][0-9]?|[1-9][0-9]|[0-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]?|[1-9][0-9]|[0-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]?|[1-9][0-9]|[0-9])\.(25[0-5]|2[0-4][0-9]|1[0-9][0-9]?|[1-9][0-9]|[0-9])$"
IPNETMASK_REGEX="^(((255\.){3}(255|254|252|248|240|224|192|128|0+))|((255\.){2}(255|254|252|248|240|224|192|128|0+)\.0)|((255\.)(255|254|252|248|240|224|192|128|0+)(\.0+){2})|((255|254|252|248|240|224|192|128|0+)(\.0+){3}))$"
UINT_REGEX="^[0-9]+$"
INT_REGEX="^[-]*[0-9]+$"
TIME_REGEX="^[0-9]+[d|h|m|s]$"
POSITIVE_NUM_REGEX="^[1-9][0-9]?$"
TEXT_REGEX="^\S+$"
UNIT_REGEX="[0-9]+"

IPADDRESS_REGEX_RAW="(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)"
# shellcheck source=/dev/null
source /lib/moxa/MoxaConnectionManager/script/bash-ini-parser.sh
#set -x
######################################################################
# shellcheck disable=SC2034
declare -A Modem_comment=(
    ["Interface"]="# Moxa Connection Manager (MCM) - Data Interface Configuration"
    ["connection.always-keep-alive"]="#[WAN ONLY] Enable/Disable Keepalive - Monitoir this interface connection.\n#Support an boolean value \"true/false\""
    ["profile.retry-threshold"]="#[WAN ONLY] Maximum retry of each profile. for profile based intra-failover."
    ["profile.failover-priority"]="#[WAN ONLY] Lists interface intra-failover profile priority index separated by ','"
    ["profile.wait-connected-timeout"]="#Profile maximum waiting connected timer\n#Specified in seconds"
    ["cellular.diag.collect-adv-info"]="#Allows to collect module-info when it suffers unknown disconnection.\n#Support an boolean value"
    ["Profile-1"]="\n#LAN/Manual will only use Profile-1"
    ["ip-method"]="#support value :\"IPV4/IPV6/IPV4V6\""
    ["connection.check-method"]="#support value :\"ping/check-ip-exist\""
    ["ping.timeout"]="#Setting the ping method waiting response timeout\n#Specified in seconds"
    ["connection.check-alive-interval"]="#Check connection interval while Keepalive enabled.\n#Specified in seconds"
    ["connection.check-failed-threshold"]="#Setting the check method fail threshold. default set fail 3 times as failed"
    ["gps.default.enabled"]="#Determine the default status (enabled/disabled) of GPS. Support an boolean value \"true/false\""
)

# shellcheck disable=SC2034
declare -A WiFi_comment=(
    ["Interface"]="# Moxa Connection Manager (MCM) - Data Interface Configuration"
    ["connection.always-keep-alive"]="#[WAN ONLY] Enable/Disable Keepalive - Monitoir this interface connection.\n#Support an boolean value \"true/false\""
    ["profile.retry-threshold"]="#[WAN ONLY] Maximum retry of each profile. for profile based intra-failover."
    ["profile.failover-priority"]="#[WAN ONLY] Lists interface intra-failover profile priority index separated by ','"
    ["profile.wait-connected-timeout"]="#Profile maximum waiting connected timer\n#Specified in seconds"
    ["Profile-1"]="\n#LAN/Manual will only use Profile-1"
    ["ip-method"]="#support value :\"IPV4/IPV6/IPV4V6\""
    ["ipv4-method"]="#Support value :\"static/dhcp\""
    ["ipv6-method"]="#support value :\"auto\""
    ["connection.check-method"]="#support value :\"ping/check-ip-exist\""
    ["ping.timeout"]="#Setting the ping method waiting response timeout\n#Specified in seconds"
    ["connection.check-alive-interval"]="#Check connection interval while Keepalive enabled.\n#Specified in seconds"
    ["connection.check-failed-threshold"]="#Setting the check method fail threshold. default set fail 3 times as failed"
)
# shellcheck disable=SC2034
declare -A WiFiP2P_comment=(
    ["Interface"]="# Moxa Connection Manager (MCM) - Data Interface Configuration"
    ["connection.always-keep-alive"]="#[WAN ONLY] Enable/Disable Keepalive - Monitoir this interface connection.\n#Support an boolean value \"true/false\""
    ["profile.retry-threshold"]="#[WAN ONLY] Maximum retry of each profile. for profile based intra-failover."
    ["profile.failover-priority"]="#[WAN ONLY] Lists interface intra-failover profile priority index separated by ','"
    ["profile.wait-connected-timeout"]="#Profile maximum waiting connected timer\n#Specified in seconds"
    ["Profile-1"]="\n#LAN will only use Profile-1"
    ["ip-method"]="#support value :\"IPV4\""
    ["ipv4-method"]="#Support value :\"static\""
)
# shellcheck disable=SC2034
declare -A Ethernet_comment=(
    ["Interface"]="# Moxa Connection Manager (MCM) - Data Interface Configuration"
    ["connection.always-keep-alive"]="#[WAN ONLY] Enable/Disable Keepalive - Monitoir this interface connection.\n#Support an boolean value \"true/false\""
    ["profile.retry-threshold"]="#[WAN ONLY] Maximum retry of each profile. for profile based intra-failover."
    ["profile.failover-priority"]="#[WAN ONLY] Lists interface intra-failover profile priority index separated by ','"
    ["profile.wait-connected-timeout"]="#Profile maximum waiting connected timer\n#Specified in seconds"
    ["dhcp-server.enabled"]="#[LAN ONLY] Support an boolean value \"true/false\""
    ["dhcp-server.ip-lease-range"]="#[LAN ONLY] dhcp server start/end address separated by ','"
    ["Profile-1"]="\n#LAN/Manual will only use Profile-1"
    ["ip-method"]="#support value :\"IPV4/IPV6/IPV4V6\""
    ["ipv4-method"]="#Support value :\"static/link-local/dhc\""
    ["ipv6-method"]="#support value :\"auto\""
    ["connection.check-method"]="#support value :\"ping/check-ip-exist\""
    ["ping.timeout"]="#Setting the ping method waiting response timeout\n#Specified in seconds"
    ["connection.check-alive-interval"]="#Check connection interval while Keepalive enabled.\n#Specified in seconds"
    ["connection.check-failed-threshold"]="#Setting the check method fail threshold. default set fail 3 times as failed"
)
# shellcheck disable=SC2034
declare -A Main_comment=(
    ["Main"]="\n#Moxa Connection Manager (MCM) - Main Configuration"
    ["log-level"]="#Support value=ERR/WARN/INFO/DEBUG/TRACE"
    ["manual.interface.list"]="#List of interface(s) only connect 1 time, without auto connection recovery\n#Lists interface name separated by ','"
    ["failover.interface.priority"]="#List of WAN interfaces manage by MCM with auto connection recovery\n#Lists Keepalive default route priority interface name separated by ','.\n#below interface list will be bring up."
    ["failback.enabled"]="#Support an boolean value \"true/false\""
    ["failback.check-interval"]="#IF enabled failback feature this timer will start at failover happen.\n#setting to '0' as seamless failback."
    ["failover.seamless"]="#When enabled, interface will immediately switch to other currently connected interfaces \n#according to the priority to reduce the disconnection time wheh it is disconnected"
    ["bridge.interface.list"]="#List of LAN interfaces manage by MCM for DHCP server setup\n#Lists interface name separated by ','\n#Below interface list will be bridged into 'mcm-br0'"
    ["standalone.interface.list"]="#Lists interface name separated by ','"
    ["dhcp-server.enabled"]="#[bridge only]\n#Support an boolean value \"true/false\""
    ["dhcp-server.ip-lease-range"]="#dhcp server start/end address separated by ','"
    ["global.dns.servers"]="#Support global DNS server"
)
########################################################################
dialog_exit() {
    local result
    yesno_box "Do you want to save configure files"
    result=$?

    if [ "$result" = "$OK" ]; then
        save_configure_files

        yesno_box "Do you want to reload configure files"
        result=$?
        if [ "$result" = "$OK" ]; then
            fail_message=$(mx-connect-mgmt reload 2>&1 | grep failed)
            if [ -z "${fail_message}" ]; then
                msg_box "Reload Successfully"
            else
                msg_box "${fail_message}"
            fi
        fi

    else
        remove_modify_file
    fi

    exit 0
}

dbg() {
    if [[ ${DEBUG} == "y" ]]; then
        echo "$@"
    fi
}

temporary_interface_configure_file() {
    local interface=$1
    [ -z "${interface}" ] && return

    local temp_conf_file=${INTERFACE_CONF_PATH}/${interface}.conf.bk
    dbg "Write Interface Configure FILE:${temp_conf_file}"
    cfg_writer >"${temp_conf_file}"
}

save_configure_files() {

    ####### Interface Configure File ########
    if [ -n "${INTERFACE}" ]; then
        temporary_interface_configure_file "${INTERFACE}"
    fi

    local interface_files=
    interface_files=$(ls ${INTERFACE_CONF_PATH}/*.bk 2>/dev/null)

    for conf_file in ${interface_files}; do
        local current_conf_file=${conf_file:0:-3}

        dbg "current_conf_file ${current_conf_file}"
        mv "${conf_file}" "${current_conf_file}"
    done

    ##update seamless failover
    for interface in "${!INTERFACE_LIST[@]}"; do
        interface_conf_file=${INTERFACE_CONF_PATH}/${interface}.conf

        if [ "${INTERFACE_AL_KEEPALIVE_LIST[${interface}]}" = "true" ]; then
            sed -i "s|connection.always-keep-alive=.*|connection.always-keep-alive=true|g" "${interface_conf_file}"
        else
            sed -i "s|connection.always-keep-alive=.*|connection.always-keep-alive=false|g" "${interface_conf_file}"
        fi
    done

    INTERFACE=
    INTERFACE_DEVICE_TYPE=
    INTERFACE_PROFILE_LIST=
    T_PROFILE=

    ####### Main Configuer File ##############
    local main_conf_file=${CONF_PATH}/MoxaConnectionManager.conf
    dbg "Main Configure file:${main_conf_file}"
    cfg_clear
    cfg_parser ${main_conf_file}

    # save main configure file
    ini_set_value Main "log-level" "${LOG_LEVEL}"
    ini_set_value Main "auto-timesync" "${AUTO_TIMESYNC}"
    ini_set_value Main "auto-timesync.interval" "${AUTO_TIMESYNC_INTERVAL}"
    ini_set_value Main "auto-timesync.interface" "${AUTO_TIMESYNC_INTERFACE}"
    ini_set_value WAN "failback.enabled" "${FAILBACK_ENABLED}"
    ini_set_value WAN "failback.check-interval" "${FAILBACK_CHECK_INTERVAL}"
    ini_set_value WAN "failover.seamless" "${FAILOVER_SEAMLESS_ENABLED}"
    ini_set_value WAN "global.dns.servers" "${GLOBAL_DNS_SERVERS}"

    ini_set_value LAN "dhcp-server.enabled" "${BRIDGE_DHCP_ENABLED}"
    ini_set_value LAN "ipv4-address" "${BRIDGE_IPV4_ADDRESS}"
    ini_set_value LAN "ipv4-netmask" "${BRIDGE_IPV4_NETMASK}"
    ini_set_value LAN "dhcp-server.ip-lease-range" "${BRIDGE_IP_RANGE}"
    ini_set_value LAN "dhcp-server.ip-lease-time" "${BRIDGE_IP_LEASE_TIME}"
    local ONSHOT_LIST_STR=
    local BRIDGE_LIST_STR=
    local LAN_LIST_STR=
    local FAILOVER_LIST_STR=

    for interface in "${!INTERFACE_LIST[@]}"; do
        local mode=${INTERFACE_LIST[${interface}]}
        case "${mode}" in
        "WAN") ;;

        "Manual")
            if [ -n "${ONSHOT_LIST_STR}" ]; then
                ONSHOT_LIST_STR+=,
            fi
            ONSHOT_LIST_STR+=${interface}
            ;;
        "LAN")
            if [ -n "${LAN_LIST_STR}" ]; then
                LAN_LIST_STR+=,
            fi
            LAN_LIST_STR+=${interface}
            ;;
        "LAN-Bridge")
            if [ -n "${BRIDGE_LIST_STR}" ]; then
                BRIDGE_LIST_STR+=,
            fi
            BRIDGE_LIST_STR+=${interface}
            ;;
        esac
    done

    for i in 0 1 2 3; do
        if [ -n "${FAILOVER_LIST[$i]}" ]; then

            if [ -n "${FAILOVER_LIST_STR}" ]; then
                FAILOVER_LIST_STR+=,
            fi
            FAILOVER_LIST_STR+=${FAILOVER_LIST[$i]}
        fi
    done

    ini_set_value Main "manual.interface.list" "${ONSHOT_LIST_STR}"
    ini_set_value LAN "bridge.interface.list" "${BRIDGE_LIST_STR}"
    ini_set_value LAN "standalone.interface.list" "${LAN_LIST_STR}"
    ini_set_value WAN "failover.interface.priority" "${FAILOVER_LIST_STR}"

    cfg_writer >${main_conf_file}
}

remove_modify_file() {
    local modify_files=
    modify_files=$(find ${CONF_PATH} -name "*.bk")
    for conf_file in ${modify_files}; do
        rm -rf "${conf_file}"
    done
}

yesno_box() {
    Input1=$1

    Msg="$Input1\n"

    dialog --title "Moxa Connection Manager" --clear \
        --backtitle "$BACKTITLE" \
        --yesno "$Msg" 10 70 \
        >/dev/tty 2>&1

    result=$?
    return $result
}

msg_box() {
    Input1=$1

    Msg="$Input1\n"

    dialog --title "Moxa Connection Manager" --clear \
        --backtitle "$BACKTITLE" \
        --msgbox "$Msg" 10 70 \
        >/dev/tty 2>&1

    result=$?
    return $result
}

help_box() {
    local text="$1"

    dialog --title "HELP" --clear \
        --backtitle "$BACKTITLE" \
        --colors \
        --msgbox "${text}" 30 100 \
        >/dev/tty 2>&1

    result=$?
    return $result
}

_radiolist() {
    local title=$1
    local text=$2
    local default=$3
    local configure_data=$4
    local handler=$5
    local select=
    local dialog_cmd=
    local index=0

    dialog_cmd="dialog --title '${title}' --clear \
        --backtitle '$BACKTITLE' \
        --cancel-label 'Exit' \
        --radiolist '${text}' 15 70 10 \
        2>&1 > /dev/tty"

    for conf in ${configure_data}; do
        if [ "${default}" = "${conf}" ]; then
            dialog_cmd+=" ${index} ${conf} on "
        else
            dialog_cmd+=" ${index} ${conf} off "
        fi
        index=$((index + 1))
    done

    dbg "${dialog_cmd}"

    select=$(eval "${dialog_cmd}")
    result=$?

    if [ $result -eq $CANCEL ]; then
        dialog_exit
        exit 0
    elif [ $result -eq $ESC ]; then
        return 0
    elif [ $result -ne $OK ]; then
        echo "No Know Key"
        exit 1
    fi

    dbg "Select:${select}"
    eval "${handler} ${select}"
}
_radiolist_long() {
    local title=$1
    local text=$2
    local default=$3
    local configure_data=$4
    local handler=$5
    local select=
    local dialog_cmd=
    local index=0

    dialog_cmd="dialog --title '${title}' --clear \
        --backtitle '$BACKTITLE' \
        --cancel-label 'Exit' \
        --radiolist '${text}' 30 70 10 \
        2>&1 > /dev/tty"

    for conf in ${configure_data}; do
        if [ "${default}" = "${conf}" ]; then
            dialog_cmd+=" ${index} ${conf} on "
        else
            dialog_cmd+=" ${index} ${conf} off "
        fi
        index=$((index + 1))
    done

    dbg "${dialog_cmd}"

    select=$(eval "${dialog_cmd}")
    result=$?

    if [ $result -eq $CANCEL ]; then
        dialog_exit
        exit 0
    elif [ $result -eq $ESC ]; then
        return 0
    elif [ $result -ne $OK ]; then
        echo "No Know Key"
        exit 1
    fi

    dbg "Select:${select}"
    eval "${handler} ${select}"
}

_interface_mode_select_handler() {
    local select=$1
    local new_mode=
    local old_mode=
    local device_type=${INTERFACE_DEVICE_TYPE}

    old_mode=${INTERFACE_LIST[${INTERFACE}]}
    if [ "${device_type}" = "Ethernet" ]; then
        case "${select}" in
        0) #WAN
            new_mode="WAN"
            ;;
        1) #LAN
            new_mode="LAN"
            ;;
        2) #LAN-Bridge
            new_mode="LAN-Bridge"
            ;;
        3) #Manual
            new_mode="Manual"
            ;;
        4) #Umanaged
            new_mode="None"
            ;;
        esac
    elif [ "${device_type}" = "WiFi" ] || [ "${device_type}" = "Modem" ]; then
        case "${select}" in
        0) #WAN
            new_mode="WAN"
            ;;
        1) #Manual
            new_mode="Manual"
            ;;
        2) #Umanaged
            new_mode="None"
            ;;
        esac
    elif [ "${device_type}" = "WiFi-P2P" ]; then
        case "${select}" in
        0) #LAN
            new_mode="LAN"
            ;;

        1) #Umanaged
            new_mode="None"
            ;;
        esac
    fi

    if [ "${old_mode}" != ${new_mode} ]; then
        if [ "${old_mode}" = "WAN" ]; then
            for i in 0 1 2 3; do
                if [ "${FAILOVER_LIST[$i]}" = "${INTERFACE}" ]; then
                    FAILOVER_LIST[$i]=""
                fi
            done
            INTERFACE_AL_KEEPALIVE_LIST[${INTERFACE}]="false"
        fi
        if [ "${new_mode}" = "WAN" ]; then
            for i in 0 1 2 3; do
                if [ -z "${FAILOVER_LIST[$i]}" ]; then
                    FAILOVER_LIST[$i]=${INTERFACE}
                    break
                fi
            done
        fi

        INTERFACE_LIST[${INTERFACE}]=${new_mode}

        #seamless feature
        local seamless_failover_interfaces_count=0
        for i in 0 1 2 3; do
            if [ -n "${FAILOVER_LIST[$i]}" ]; then
                seamless_failover_interfaces_count=$((seamless_failover_interfaces_count + 1))
            fi
        done

        if [ ${seamless_failover_interfaces_count} -le 1 ]; then
            FAILOVER_SEAMLESS_ENABLED=false
            _disable_all_wan_keepalive
        fi

        if [ "${FAILOVER_SEAMLESS_ENABLED}" == "true" ]; then
            local result
            yesno_box "Do you want to change seamless failover interfaces"
            result=$?

            if [ "$result" = "$OK" ]; then
                dialog_seamless_interfaces
            fi
        fi
    fi
}

dialog_interface_mode_select() {
    local default=$1
    local device_type=${INTERFACE_DEVICE_TYPE}
    local select_option=
    local text=

    text="Select an Option:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    case "${device_type}" in
    "Ethernet")
        select_option="WAN LAN LAN-Bridge Manual None"
        ;;
    "WiFi")
        select_option="WAN Manual None"
        ;;
    "WiFi-P2P")
        select_option="LAN None"
        ;;
    "Modem")
        select_option="WAN Manual None"
        ;;
    esac

    _radiolist "Select Network Type" "${text}" "${default}" "${select_option}" "_interface_mode_select_handler"
}

_interface_keepalive_enabled_handler() {
    local select=$1
    local key="connection.always-keep-alive"
    local val=
    case "${select}" in
    0) #Enable
        val="true"
        ;;
    1) #Disable
        val="false"
        ;;
    esac
    ini_set_value Interface "${key}" "${val}"
}

dialog_interface_keepalive_enabled() {
    local default=$1
    local text=

    text="Select an Option:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "Enable/Disable Keep-alive" "${text}" "${default}" "Enable Disable" "_interface_keepalive_enabled_handler"
}

_interface_gps_enabled_handler() {
    local select=$1
    local key="gps.default.enabled"
    local val=
    case "${select}" in
    0) #Enable
        val="true"
        ;;
    1) #Disable
        val="false"
        ;;
    esac
    ini_set_value Interface "${key}" "${val}"
}

dialog_gps_default_enabled() {
    local default=$1
    local text=

    text="Select an Option:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "Enable/Disable GPS" "${text}" "${default}" "Enable Disable" "_interface_gps_enabled_handler"
}

_interface_gps_antenna_handler() {
    local select=$1
    local key="gps.antenna.type"
    local val=
    case "${select}" in
    0) #Enable
        val="active"
        ;;
    1) #Disable
        val="passive"
        ;;
    esac
    ini_set_value Interface "${key}" "${val}"
}

dialog_gps_antenna_type() {
    local default=$1
    local text=

    text="Select an Option:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "GPS Antenna Type" "${text}" "${default}" "active passive" "_interface_gps_antenna_handler"
}

_interface_dhcp_server_enabled_handler() {
    local select=$1
    local key="dhcp-server.enabled"
    local val=
    case "${select}" in
    0) #Enable
        val="true"
        ;;
    1) #Disable
        val="false"
        ;;
    esac
    DHCP_SERVER=${val}
    ini_set_value Interface "${key}" "${val}"
}

dialog_interface_dhcp_server_enabled() {
    local default=$1
    local text=

    text="Select an Option:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "Enable/Disable DHCP Server" "${text}" "${default}" "Enable Disable" "_interface_dhcp_server_enabled_handler"
}

_interface_bridge_dhcp_server_enabled_handler() {
    local select=$1
    local val=
    case "${select}" in
    0) #Enable
        val="true"
        ;;
    1) #Disable
        val="false"
        ;;
    esac
    BRIDGE_DHCP_ENABLED=${val}
}

dialog_interface_bridge_dhcp_server_enabled() {
    local default=$1
    local text=

    text="Select an Option:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "Enable/Disable DHCP Server" "${text}" "${default}" "Enable Disable" "_interface_bridge_dhcp_server_enabled_handler"
}

_configure_failback_enabled_handler() {
    local select=$1
    local val=

    local val=
    case "${select}" in
    0) #Enable
        val="true"
        ;;
    esac

    FAILBACK_ENABLED=${val}
}

dialog_configure_failback_enabled() {
    local default=$1
    local title="Enable/Disable Failback"
    local text="When enabled, the backup connection will automatically failback to the higher priority connection when it became available again \
    \n\nSelect an Option:\
    \n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "${title}" "${text}" "${default}" "Enable" "_configure_failback_enabled_handler"
}

_seamless_interface_keepalive_enabled_handler() {
    local select=$1
    local key="connection.always-keep-alive"
    local val=
    case "${select}" in
    0) #Enable
        val="true"
        ;;
    1) #Disable
        val="false"
        ;;
    esac

    INTERFACE_AL_KEEPALIVE_LIST[${SEAMLESS_INTERFACE}]=$val
}

dialog_seamless_interface_keepalive_enabled() {
    local default=$1
    local seamless_interface=$2
    local text=
    SEAMLESS_INTERFACE=$seamless_interface
    text="Select an Option:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "Enable/Disable Keep-alive for seamless failover " "${text}" "${default}" "Enable Disable" "_seamless_interface_keepalive_enabled_handler"
}

_disable_all_wan_keepalive() {
    for i in 0 1 2 3; do
        if [ -n "${FAILOVER_LIST[$i]}" ]; then
            local interface=${FAILOVER_LIST[$i]}
            INTERFACE_AL_KEEPALIVE_LIST[${interface}]="false"
        fi
    done
}

dialog_seamless_interfaces() {
    local title="Configure Seamless Failover Interfaces"
    local basic_dialog_cmd=
    local dialog_cmd=
    local dialog_window_conf=
    local index=1
    local key=

    declare -A dialog_window_conf

    basic_dialog_cmd="dialog --title '${title}' --clear \
            --backtitle '$BACKTITLE' \
            --cancel-label 'Exit' \
            --colors \
            --menu '\nSpace to toggle, Enter to confirm, ESC to go to the previous page' 30 80 30 \
            2>&1 > /dev/tty"

    while true; do
        dialog_cmd=${basic_dialog_cmd}
        index=1
        #skip priority 1 interface , because alway reconnect
        for i in 1 2 3; do
            if [ -n "${FAILOVER_LIST[$i]}" ]; then
                local interface=${FAILOVER_LIST[$i]}

                if [ "${INTERFACE_AL_KEEPALIVE_LIST[${interface}]}" = "true" ]; then
                    dialog_cmd+=" ' ${index}' '${FAILOVER_LIST[$i]} [Enabled]'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_seamless_interface_keepalive_enabled Enable ${interface}'"
                else
                    dialog_cmd+=" ' ${index}' '${FAILOVER_LIST[$i]} [Disabled]'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_seamless_interface_keepalive_enabled Disable ${interface}'"
                fi
                index=$((index + 1))
            fi
        done

        dbg "${dialog_cmd}"
        local next_dialog_window=

        select=$(eval "${dialog_cmd}")
        result=$?

        if [ $result -eq $CANCEL ]; then
            dialog_exit
            exit 0
        elif [ $result -eq $ESC ]; then
            return 0
        elif [ $result -ne $OK ]; then
            echo "No Know Key"
            exit 1
        fi

        dbg "Select:${select}"
        next_dialog_window=${dialog_window_conf[" ${select} "]}
        dbg "Next Window Name:${next_dialog_window}"

        if [ -n "${next_dialog_window}" ]; then
            eval "${next_dialog_window}"
        fi
    done
}

_configure_seamless_failover_handler() {
    local select=$1
    local val=

    case "${select}" in
    0) #Enable
        val="true"
        ;;
    1) #Disable
        val="false"
        ;;
    esac

    FAILOVER_SEAMLESS_ENABLED=${val}
    if [ "${val}" == "true" ]; then
        dialog_seamless_interfaces
    else
        _disable_all_wan_keepalive
    fi
}

dialog_configure_seamless_failover_enabled() {
    local default=$1
    local title="Enable/Disable Seamless Failback"
    local text="Disabled: \
    \nWhen the primary interface connection becomes unavailable, MCM will attempt to establish a connection using all the pre-configured profiles before switching to the backup interface. There will be a period of downtime during the failback process. \
    \n\nEnabled: \
    \nWhen the primary interface becomes unavailable, MCM will seamlessly switch the default gateway to the already connected backup interface, avoiding any downtime during the failback process. However, it is essential to be aware using ping as the keep-alive method for the backup interface might incur some data costs. \
    \n\nSelect an Option:\
    \n(Space to toggle, Enter to confirm, ESC to the previous page)"

    local failover_interface=0

    for i in 0 1 2 3; do
        if [ -n "${FAILOVER_LIST[$i]}" ]; then
            failover_interface=$((failover_interface + 1))
        fi
    done

    #no any failver interface
    if [ "${failover_interface}" == "0" ]; then
        msg_box "Failover priority list does not contain any interfaces. To enable seamless failover, you must set at least two interfaces in the priority list."
        return
    fi

    if [ "${failover_interface}" == "1" ]; then
        msg_box "Failover priority list contains only one interface. To enable seamless failover, you must set at least two interfaces in the priority list."
        return
    fi

    _radiolist_long "${title}" "${text}" "${default}" "Enable Disable" "_configure_seamless_failover_handler"
}

_configure_log_level_handler() {
    local select=$1
    local val=

    case "${select}" in
    0) #ERR
        val="ERR"
        ;;
    1) #WARN
        val="WARN"
        ;;
    2) #INFO
        val="INFO"
        ;;
    3) #DEBUG
        val="DEBUG"
        ;;
    4) #TRACE
        val="TRACE"
        ;;
    esac

    LOG_LEVEL=${val}
}

dialog_configure_log_level() {
    local default=$1
    local title="Configure Moxa Connection Manager (MCM) Log Leve"
    local text="Select Log Level\n\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "${title}" "${text}" "${default}" " ERR WARN INFO DEBUG TRACE" "_configure_log_level_handler"
}

_configure_auto_timesync_handler() {
    local select=$1
    local val=

    case "${select}" in
    0)
        val="Disabled"
        ;;
    1)
        val="GPS"
        ;;
    2)
        val="Chrony"
        ;;
    3)
        val="Cellular"
        ;;
    esac
    AUTO_TIMESYNC=${val}
}

dialog_configure_auto_timesync() {
    local default=$1
    local title="Configure Auto Synchronization"
    local text="Select Auto Timesync Method\n\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    _radiolist "${title}" "${text}" "${default}" " Disabled GPS Chrony Cellular" "_configure_auto_timesync_handler"
}

_auto_timesync_interval_handler() {
    local select=$2
    local key=$1

    AUTO_TIMESYNC_INTERVAL="${select}"
}

dialog_configure_auto_timesync_interval() {
    local title="Configure Auto Timesync Interval"
    local default=$1
    local text=
    local regex=${UINT_REGEX}
    local key="auto-timesync.interval"
    text="Input a Number: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_auto_timesync_interval_handler ${key} "
}

_auto_timesync_interface_handler() {
    local select=$1
    local val=

    #Cellular interface name is fixed
    case "${select}" in
    0)
        val="Cellular1"
        ;;
    1)
        val="Cellular2"
        ;;
    2)
        val="Cellular3"
        ;;
    3)
        val="Cellular4"
        ;;
    esac
    AUTO_TIMESYNC_INTERFACE="${val}"
}

dialog_configure_auto_timesync_interface() {
    local title="Configure Auto Timesync Interface"
    local default=$1
    local text=
    local interface_list
    local select_option

    text="Select an Option:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)"

    interface_list=$(mx-connect-mgmt ls | grep InterfaceName | cut -d ":" -f 2)
    for interface in ${interface_list}; do
        local type
        type=$(mx-connect-mgmt nwk_info "${interface}" | grep "Device Type" | cut -d ':' -f 2 | tr -d ' ')

        if [ "${type}" = "Modem" ]; then
            select_option="${select_option} ${interface}"
        fi
    done

    _radiolist "${title}" "${text}" "${default}" "${select_option}" "_auto_timesync_interface_handler"
}

_inputbox() {
    local title=$1
    local text=$2
    local default=$3
    local select=
    local dialog_cmd=
    local index=0
    local handler=$4

    dialog_cmd="dialog --title '${title}' \
        --backtitle '$BACKTITLE' \
        --cancel-label 'Exit' \
        --inputbox '${text}' 10 110 '${default}' \
        2>&1 >/dev/tty"
    dbg "$dialog_cmd"

    select=$(eval "$dialog_cmd")
    result=$?

    if [ "$result" -eq "$CANCEL" ]; then
        dialog_exit
        exit 0
    elif [ "$result" -eq "$ESC" ]; then
        return 0
    elif [ "$result" -ne "$OK" ]; then
        echo "No Known Key"
        exit 1
    fi

    #dbg "Select: $select"
    eval "${handler} '${select}'"
}
_inputbox_check() {
    local title=$1
    local text=$2
    local default=$3
    local select=
    local dialog_cmd=
    local index=0
    local regex=$4
    local handler=$5
    local usage=$6

    if [ -z "${usage}" ]; then
        usage="Format is incorrect"
    fi

    while true; do
        dialog_cmd="dialog --title '${title}' \
            --backtitle '$BACKTITLE' \
            --cancel-label 'Exit' \
            --inputbox  '${text}'\
            20 110 '${default}' \
            2>&1 >/dev/tty"
        dbg "${dialog_cmd}"

        select=$(eval "${dialog_cmd}")
        result=$?

        if [ $result -eq $CANCEL ]; then
            dialog_exit
            exit 0
        elif [ $result -eq $ESC ]; then
            return 0
        elif [ $result -ne $OK ]; then
            echo "No Know Key"
            exit 1
        fi

        dbg "Select:${select}"

        if [[ ${select} =~ ${regex} ]]; then
            break
        else
            msg_box "${usage}"
        fi
    done

    dbg "Select:${select}"
    eval "${handler} \"${select}\""
}

dialog_dhcp_ip_range() {
    local title="DHCP Server IP Range"
    local text="Input DHCP address Range:"
    local default=$1
    local select=
    local dialog_cmd=
    local start_ip
    local end_ip
    local key="dhcp-server.ip-lease-range"

    start_ip=$(echo "${default}" | cut -d ',' -f 1)
    end_ip=$(echo "${default}" | cut -d ',' -f 2)

    while true; do
        dialog_cmd="dialog --title '${title}' \
            --backtitle '$BACKTITLE' \
            --cancel-label 'Exit' \
            --form '${text}' 10 50 3 \
            'Start:' 1 1 ${start_ip} 1 15 25 0 \
            'End:' 2 1 ${end_ip} 2 15 25 0 \
            2>&1 >/dev/tty"

        dbg "${dialog_cmd}"

        select=$(eval "${dialog_cmd}")
        result=$?

        if [ $result -eq $CANCEL ]; then
            dialog_exit
            exit 0
        elif [ $result -eq $ESC ]; then
            return 0
        elif [ $result -ne $OK ]; then
            echo "No Know Key"
            exit 1
        fi

        dbg "Select:${select}"

        # shellcheck disable=SC2206
        value=(${select})

        if [[ ${value[0]} =~ ${IPADDRESS_REGEX} ]] && [[ ${value[1]} =~ ${IPADDRESS_REGEX} ]]; then
            break
        else
            msg_box "Format is incorrect"
        fi
    done

    dbg "Select:${select}"

    new_data=${value[0]},${value[1]}
    ini_set_value Interface ${key} "${new_data}"
}

dialog_bridge_dhcp_ip_range() {
    local title="DHCP Server IP Range"
    local text="Input DHCP address Range:"
    local default=$1
    local select=
    local dialog_cmd=
    local start_ip
    local end_ip
    local key="dhcp-server.ip-lease-range"

    start_ip=$(echo "${default}" | cut -d ',' -f 1)
    end_ip=$(echo "${default}" | cut -d ',' -f 2)

    while true; do
        dialog_cmd="dialog --title '${title}' \
            --backtitle '$BACKTITLE' \
            --cancel-label 'Exit' \
            --form '${text}' 10 50 3 \
            'Start:' 1 1 ${start_ip} 1 15 25 0 \
            'End:' 2 1 ${end_ip} 2 15 25 0 \
            2>&1 >/dev/tty"

        dbg "${dialog_cmd}"

        select=$(eval "${dialog_cmd}")
        result=$?

        if [ $result -eq $CANCEL ]; then
            dialog_exit
            exit 0
        elif [ $result -eq $ESC ]; then
            return 0
        elif [ $result -ne $OK ]; then
            echo "No Know Key"
            exit 1
        fi

        dbg "Select:${select}"

        # shellcheck disable=SC2206
        value=(${select})

        if [[ ${value[0]} =~ ${IPADDRESS_REGEX} ]] && [[ ${value[1]} =~ ${IPADDRESS_REGEX} ]]; then
            break
        else
            msg_box "Format is incorrect"
        fi
    done

    dbg "Select:${select}"

    new_data=${value[0]},${value[1]}
    BRIDGE_IP_RANGE=${new_data}
}

_global_dns_handler() {
    local select=$2
    local key=$1

    GLOBAL_DNS_SERVERS="${select}"
}

dialog_global_dns_servers() {
    local title=
    local default=$1
    local text=
    local regex="${IPADDRESS_REGEX}|^${IPADDRESS_REGEX_RAW}(\s*\,\s*${IPADDRESS_REGEX_RAW})+$| ^$"
    local key="global.dns.servers"

    text="Input DNS Server Address(es) \
    \n\nIf the DHCP servers that the Moxa computer connects to do not provide DNS server information in their configuration, you may manually configure the DNS address(es) below. Please use commas to separate multiple IP addresses. For example: 8.8.8.8,1.1.1.1 
    \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_global_dns_handler ${key} "
}

_interface_dhcp_lease_time_handler() {
    local select=$1
    local key="dhcp-server.ip-lease-time"
    ini_set_value Interface "${key}" "${select}"
}

dialog_interface_dhcp_lease_time() {
    local title="DHCP Server IP Lease Time"
    local default=$1
    local text=
    local regex=${TIME_REGEX}

    text="Input Least Time: \
          \n\nThe time unit can be d(day), h(hours) or M(minute). e.g., 12h, 720M, 30d
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_interface_dhcp_lease_time_handler"
}

_interface_bridge_dhcp_lease_time_handler() {
    local select=$1
    BRIDGE_IP_LEASE_TIME=${select}
}

dialog_interface_bridge_dhcp_lease_time() {
    local title="DHCP Server IP Lease Time"
    local default=$1
    local text=
    local regex=${TIME_REGEX}

    text="Input Least Time: \
          \n\nThe time unit can be d(day), h(hours) or M(minute). e.g., 12h, 720M, 30d
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_interface_bridge_dhcp_lease_time_handler"
}

_interface_profile_priority_handler() {
    local select=$1
    local key="profile.failover-priority"
    ini_set_value Interface "${key}" "${select}"
}

PROFILE_PRIORITY_REGEX="^[0-9]?[0-9]*(,[0-9][0-9]*)*$"
dialog_interface_profile_priority() {
    local default=$1
    local title="${INTERFACE} Network Profile Priority"
    local text=
    local regex=${PROFILE_PRIORITY_REGEX}

    text="Input Network Profile Number in Priority Order (separate by comma): \
          \n\nFor example, enter 1,3 means MCM will use Profile-1 to connect and \
          failover to Profile-3 when Profile-1 cannot connect or become unavailable \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_interface_profile_priority_handler"
}

_interface_retry_threshold_handler() {
    local select=$1
    local key="profile.retry-threshold"
    ini_set_value Interface "${key}" "${select}"
}

dialog_interface_retry_threshold() {
    local title="${INTERFACE} Network Profile Retry Threshold"
    local default=$1
    local text=
    local regex=${POSITIVE_NUM_REGEX}

    text="Input a Number: \
          \n\nThis value determine the maximum attemps MCM will try \
           to connect using a network profile before failover to the next profile in priority list \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_interface_retry_threshold_handler"
}

_interface_profile_timeout_handler() {
    local select=$1
    local key="profile.wait-connected-timeout"
    ini_set_value Interface "${key}" "${select}"
}

dialog_interface_profile_timeout() {
    local title="${INTERFACE} Network Profile Timeout"
    local default=$1
    local text=
    local regex=${UINT_REGEX}

    text="Input a Number: \
          \n\nThis value (in seconds) deteremines the maximum time MCM will try to connect using a network profile before determine the connection is not available \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_interface_profile_timeout_handler"
}

_interface_remove_profile_handler() {
    local select=$1
    local profile_name=Profile-${select}
    local trasform_profile_name=
    trasform_profile_name=$(cfg_transform_key "${profile_name}")

    if [[ ! "${INTERFACE_PROFILE_LIST[*]}" =~ ${profile_name} ]]; then
        msg_box "${profile_name} is not existed"
        return
    fi

    if [ "${profile_name}" = "Profile-1" ]; then
        msg_box "Profile-1 should not be removed"
        return
    fi

    cfg_clear "${trasform_profile_name}"
    # shellcheck disable=SC2206
    INTERFACE_PROFILE_LIST=(${INTERFACE_PROFILE_LIST[@]/$profile_name/})
}

dialog_interface_remove_profile() {
    local title="Remove Profile"
    local default=1
    local text=
    local regex=${UINT_REGEX}

    text="Specify Profile Number: \
          \n\nFor example, enter 2 will remove Profile-2 \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_interface_remove_profile_handler"
}

_interface_add_profile_handler() {
    local select=$1
    local profile_name=Profile-${select}
    local trasform_profile_name=
    trasform_profile_name=$(cfg_transform_key "${profile_name}")

    if [[ "${INTERFACE_PROFILE_LIST[*]}" =~ ${profile_name} ]]; then
        msg_box "${profile_name} has existed"
        return
    fi

    cfg_parser ${NEW_PROFILE_PATH}/"${INTERFACE_DEVICE_TYPE}"_New_Profile.conf
    eval "$(
        echo "cfg_section_${trasform_profile_name}()"
        declare -f cfg_section_ProfileNew | tail -n +2
    )"
    cfg_clear ProfileNew

    # shellcheck disable=SC2206
    INTERFACE_PROFILE_LIST+=(${profile_name})
}

dialog_interface_add_profile() {
    local title="Add Profile"
    local default=1
    local text=
    local regex=${POSITIVE_NUM_REGEX}

    text="Specify Profile Number: \
          \n\nFor example, enter 2 to add Profile-2 \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_interface_add_profile_handler" "Profile number has to be greater than 0"
}

dialog_configure_profile_help() {
    local text=
    text="\Z4Ping Timeout\Zn
    \nThis value (in seconds) determines the maximum time MCM will try to ping the configured target host and wait for a response before determining the connection is unavailable.
    \n\n\Z4Configure Check-alive Interval\Zn
    \nThis value (in seconds) determines how often MCM check the connection is alive
    \n\n\Z4Configure Check-alive Retry Threshold\Zn
    \nThis value determines the maximum attempts MCM will check the connection status before concluding the connection is not available
    "
    help_box "${text}"
}

_configure_ethernet_dhcp_server_enable_ipv4_method() {

    local select=$0
    local val="static"
    local key="ipv4-method"

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

_configure_ethernet_ipv4_method() {

    local select=$1
    local val=
    local key="ipv4-method"

    case "${select}" in
    0)
        val="static"
        ;;
    1)
        val="dhcp"
        ;;
    2)
        val="link-local"
        ;;
    esac

    if [ "${val}" = "link-local" ]; then
        ini_set_value "${T_PROFILE}" "connection.check-method" "check-ip-exist"
    fi

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

_configure_ipv4_method() {
    local select=$1
    local val=
    local key="ipv4-method"

    case "${select}" in
    0)
        val="static"
        ;;
    1)
        val="dhcp"
        ;;
    esac

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_ethernet_ipv4_method() {
    local default=$1
    local title="Configure IPv4 Method"
    local text="Select  IPv4 Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    local dhcp_server=${DHCP_SERVER}
    if [ "${dhcp_server}" = "true" ]; then
        _radiolist "${title}" "${text}" "${default}" "static" "_configure_ethernet_dhcp_server_enable_ipv4_method"
    else
        _radiolist "${title}" "${text}" "${default}" "static dhcp link-local" "_configure_ethernet_ipv4_method"
    fi
}

_configure_wifi_p2p_ipv4_method() {
    local select=$1
    local val=
    local key="ipv4-method"

    case "${select}" in
    0)
        val="static"
        ;;
    esac

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_wifi_p2p_ipv4_method() {
    local default=$1
    local title="Configure IPv4 Method"
    local text="Select  IPv4 Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"
    _radiolist "${title}" "${text}" "${default}" "static" "_configure_wifi_p2p_ipv4_method"
}

dialog_profile_ipv4_method() {
    local default=$1
    local title="Configure IPv4 Method"
    local text="Select  IPv4 Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    _radiolist "${title}" "${text}" "${default}" "static dhcp" "_configure_ipv4_method"
}

_configure_ipv6_method() {
    local select=$1
    local val=
    local key="ipv6-method"

    case "${select}" in
    0)
        val="auto"
        ;;
    esac

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_ipv6_method() {
    local default=$1
    local title="Configure IPv6 Method"
    local text="Select  IPv4 Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    _radiolist "${title}" "${text}" "${default}" "auto" "_configure_ipv6_method"
}

_configure_ethernet_check_method() {
    local select=$1
    local val=
    local key="connection.check-method"

    case "${select}" in
    0)
        val="ping"
        ;;
    1)
        val="check-ip-exist"
        ;;
    esac

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

_configure_ethernet_link_local_check_method() {
    local select=$1
    local val=
    local key="connection.check-method"

    case "${select}" in
    0)
        val="check-ip-exist"
        ;;
    esac

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

_configure_check_method() {
    local select=$1
    local val=
    local key="connection.check-method"

    case "${select}" in
    0)
        val="ping"
        ;;
    1)
        val="check-ip-exist"
        ;;
    2)
        val="ping-signalmonitor"
        ;;
    3)
        val="check-ip-exist-signalmonitor"
        ;;
    esac

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_ethernet_check_method() {
    local default=$1
    local title="Configure Connection Check-alive Method"
    local text="Select Check Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    local key="ipv4-method"
    local val=
    val=$(ini_get_value ${key})
    local ip_method

    key="ip-method"
    ip_method=$(ini_get_value ${key})

    if [ "${val}" = "link-local" ] && [ "${ip_method}" = "IPV4" ]; then
        _radiolist "${title}" "${text}" "${default}" "check-ip-exist" "_configure_ethernet_link_local_check_method"
    else
        _radiolist "${title}" "${text}" "${default}" "ping check-ip-exist" "_configure_ethernet_check_method"
    fi
}

dialog_profile_check_method() {
    local default=$1
    local title="Configure Connection Check-alive Method"
    local text="Select Check Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    _radiolist "${title}" "${text}" "${default}" "ping check-ip-exist ping-signalmonitor check-ip-exist-signalmonitor" "_configure_check_method"
}

_configure_signal_monitor_wifi_thres_type() {
    local select=$1
    local val=
    local key="signal.monitor-threshold-type"

    case "${select}" in
    0)
        val="SIGNAL"
        ;;
    esac

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_signal_monitor_wifi_threshold_type() {
    local default=$1
    local title="Configure Signal Monitor Thresold Type"
    local text="Select Check Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    _radiolist "${title}" "${text}" "${default}" "SIGNAL " "_configure_signal_monitor_wifi_thres_type"
}

_configure_signal_monitor_cell_thres_type() {
    local select=$1
    local val=
    local key="signal.monitor-threshold-type"

    case "${select}" in
    0)
        val="RSRP"
        ;;
    1)
        val="SINR"
        ;;
    2)
        val="SIGNAL-STRENGTH"
        ;;
    esac
    SIGNAL_MONITOR_THR_TYPE="${val}"
    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_signal_monitor_cell_threshold_type() {
    local default=$1
    local title="Configure Signal Monitor Thresold Type"
    local text="Select Check Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    _radiolist "${title}" "${text}" "${default}" "RSRP SINR SIGNAL-STRENGTH " "_configure_signal_monitor_cell_thres_type"
}

_configure_ip_method() {
    local select=$1
    local val=
    local key="ip-method"

    case "${select}" in
    0)
        val="IPV4"
        ;;
    1)
        val="IPV6"
        ;;
    2)
        val="IPV4V6"
        ;;
    esac

    local ipv4_method_key="ipv4-method"
    local ipv4_method=
    ipv4_method=$(ini_get_value ${ipv4_method_key})

    if [ "${INTERFACE_DEVICE_TYPE}" = "Ethernet" ] && [ "${val}" = "IPV4" ] && [ "${ipv4_method}" = "link-local" ]; then
        ini_set_value "${T_PROFILE}" "connection.check-method" "check-ip-exist"
    fi

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_ip_method() {
    local default=$1
    local title="Configure IP Method"
    local text="Select Check Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    _radiolist "${title}" "${text}" "${default}" "IPV4 IPV6 IPV4V6" "_configure_ip_method"
}

_configure_wifi_p2p_ip_method() {
    local select=$1
    local val=
    local key="ip-method"

    case "${select}" in
    0)
        val="IPV4"
        ;;
    esac

    local ipv4_method_key="ipv4-method"
    local ipv4_method=
    ipv4_method=$(ini_get_value "${ipv4_method_key}")

    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_wifi_p2p_ip_method() {
    local default=$1
    local title="Configure IP Method"
    local text="Select Check Method\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"

    _radiolist "${title}" "${text}" "${default}" "IPV4" "_configure_wifi_p2p_ip_method"
}

_profile_inputbox_handler() {
    local select=${2}
    local key=$1

    ini_set_value "${T_PROFILE}" "${key}" "${select}"
}

_bridge_ipv4_address() {
    local select=$1
    BRIDGE_IPV4_ADDRESS=${select}
}

dialog_bridge_ipv4_address() {
    local title="IPv4 Address"
    local default=$1
    local text=
    local regex=${IPADDRESS_REGEX}
    text="Input IPv4 Address: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_bridge_ipv4_address "
}

dialog_profile_ipv4_address() {
    local title="IPv4 Address"
    local default=$1
    local text=
    local regex=${IPADDRESS_REGEX}
    local key="ipv4-address"
    text="Input IPv4 Address: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "

}

_bridge_ipv4_netmask() {
    local select=$1
    BRIDGE_IPV4_NETMASK=${select}
}

dialog_bridge_ipv4_netmask() {
    local title="IPv4 Netmask"
    local default=$1
    local text=
    local regex=${IPNETMASK_REGEX}
    text="Input IPv4 Netmask: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_bridge_ipv4_netmask "
}

dialog_profile_ipv4_netmask() {
    local title="IPv4 Netmask"
    local default=$1
    local text=
    local regex=${IPNETMASK_REGEX}
    local key="ipv4-netmask"
    text="Input IPv4 Netmask: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_ipv4_gateway() {
    local title="IPv4 Gateway"
    local default=$1
    local text=
    local regex="${IPADDRESS_REGEX}|^$"
    local key="ipv4-gateway"
    text="Input IPv4 Gateway: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_ipv4_primary_dns() {
    local title="IPv4 Primary DNS"
    local default=$1
    local text=
    local regex="${IPADDRESS_REGEX}|^$"
    local key="primary-dns"
    text="Input IPv4 Primary DNS: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_ipv4_secondary_dns() {
    local title="IPv4 Secondary DNS"
    local default=$1
    local text=
    local regex="${IPADDRESS_REGEX}|^$"
    local key="secondary-dns"
    text="Input IPv4 Secondary DNS: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_ssid() {
    local title="SSID"
    local default=$1
    local text=
    local key="SSID"
    text="Input SSID: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox "${title}" "${text}" "${default}" "_profile_inputbox_handler ${key} "
}

dialog_profile_apn() {
    local title="APN"
    local default=$1
    local text=
    local regex="${TEXT_REGEX}|^$"
    local key="APN"
    text="Input APN: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_sim() {
    local title="SIM Slot"
    local default=$1
    local text=
    local regex=${UINT_REGEX}
    local key="SIM"
    text="Input SIM Slot: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_pin() {
    local title="PIN"
    local default=$1
    local text=
    local regex="${UINT_REGEX}|^$"
    local key="PIN"
    text="Input Pin Code: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_username() {
    local title="Username"
    local default=$1
    local text=
    local key="Username"
    text="Input Username: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox "${title}" "${text}" "${default}" "_profile_inputbox_handler ${key} "
}

dialog_profile_password() {
    local title="Password"
    local default=$1
    local text=
    local key="Password"
    text="Input Password: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox "${title}" "${text}" "${default}" "_profile_inputbox_handler ${key} "
}

_configure_modem_rat() {
    local select=$1
    local val=
    local key="RAT"
    val=${MODEM_SUPPORT_RAT_LIST[$select]}
    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_rat() {
    local default=$1
    local title="Configure Modem RAT"
    local text="Select Modem RAT\n\nSpace to toggle, Enter to confirm, ESC to go to the previous page"
    local rat_list
    rat_list="${MODEM_SUPPORT_RAT_LIST[*]}"

    _radiolist "${title}" "${text}" "${default}" "${rat_list}" "_configure_modem_rat"
}

dialog_profile_ipv4_target_host() {
    local title="IPv4 Ping Target Host"
    local default=$1
    local text=
    local regex="${IPADDRESS_REGEX}|([a-zA-Z0-9]+(-[a-zA-Z0-9]+)*\\.)+[a-zA-Z]{2,}$"
    local key="ping.ipv4-target-host"
    text="Input IPv4 address: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_ipv6_target_host() {
    local title="IPv6 Ping Target Host"
    local default=$1
    local text=
    local key="ping.ipv6-target-host"
    text="Input IPv6 address: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox "${title}" "${text}" "${default}" "_profile_inputbox_handler ${key} "
}

dialog_profile_ping_timeout() {
    local title="Ping Timeout"
    local default=$1
    local text=
    local regex=${UINT_REGEX}
    local key="ping.timeout"
    text="Input a Number: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_connection_check_alive_interval() {
    local title="Configure Check-alive Check Interval"
    local default=$1
    local text=
    local regex=${UINT_REGEX}
    local key="connection.check-alive-interval"
    text="Input a Number: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_connection_check_failed_threshold() {
    local title="Configure Check-alive Retry Threshold"
    local default=$1
    local text=
    local regex=${POSITIVE_NUM_REGEX}
    local key="connection.check-failed-threshold"
    text="Input a Number: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_connection_check_failed_threshold() {
    local title="Configure Check-alive Retry Threshold"
    local default=$1
    local text=
    local regex=${POSITIVE_NUM_REGEX}
    local key="connection.check-failed-threshold"
    text="Input a Number: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

dialog_profile_signal_monitor_threshold() {
    local title="Signal Monitor Threshold"
    local default=$1
    local text=
    local key="signal.monitor-threshold"
    local regex=${INT_REGEX}
    text="Input Signal Monitor Threshold with integer \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

profile_cell_signal_monitor_threshold_handler() {
    local select=$1
    local val=
    local key="ip-method"

    case "${select}" in
    0)
        val="NONE"
        ;;
    1)
        val="POOR"
        ;;
    2)
        val="FAIR"
        ;;
    3)
        val="GOOD"
        ;;
    4)
        val="EXCELLENT"
        ;;
    esac

    local key="signal.monitor-threshold"
    SIGNAL_MONITOR_THR=${val}
    ini_set_value "${T_PROFILE}" "${key}" "${val}"
}

dialog_profile_cell_signal_monitor_threshold() {
    local title="Signal Monitor Threshold"
    local default=$1
    local text=
    local quality_text=
    local regex=${INT_REGEX}
    local key="signal.monitor-threshold"
    text="Input Signal Monitor Threshold with integer \
          \n\nEnter to confirm, ESC to go to the previous page"
    quality_text="Select Signal Strength level for Signal Monitor Threshold  \
          \n\nEnter to confirm, ESC to go to the previous page"

    if [ "${SIGNAL_MONITOR_THR_TYPE}" = "SIGNAL-STRENGTH" ]; then
        _radiolist "${title}" "${quality_text}" "${default}" "NONE POOR FAIR GOOD EXCELLENT" "profile_cell_signal_monitor_threshold_handler"
    else
        _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
    fi
}

dialog_profile_signal_monitor_long_interval() {
    local title="Signal Monitor Long Interval"
    local default=$1
    local text=
    local regex=${UINT_REGEX}
    local key="signal.monitor-long-interval"
    text="Input an Integer Number: \
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_profile_inputbox_handler ${key} "
}

setup_level2_window() {
    local key=$1
    local text=$2
    local handler=$3
    local val=

    val=$(ini_get_value "${key}")
    dialog_cmd+=" ' ${index}' '|----------- ${text} [${val}]'"
    eval "dialog_window_conf[ ' ${index}' ]='${handler} \"${val}\"'"
    index=$((index + 1))
}

setup_level1_window() {
    local key=$1
    local text=$2
    local handler=$3
    local val=

    val=$(ini_get_value "${key}")
    dialog_cmd+=" ' ${index}' '|---- ${text} [${val}]'"
    eval "dialog_window_conf[ ' ${index}' ]='${handler} \"${val}\"'"
    index=$((index + 1))
}

setup_level0_window() {
    local key=$1
    local text=$2
    local handler=$3
    local val=

    val=$(ini_get_value "${key}")
    dialog_cmd+=" ' ${index}' '${text} [${val}]'"
    eval "dialog_window_conf[ ' ${index}' ]='${handler} \"${val}\"'"
    index=$((index + 1))
}

dialog_configure_profile() {
    local profile=$1
    local title="Configure ${interface} ${profile}"
    local basic_dialog_cmd=
    local dialog_cmd=
    local index=1
    local key=
    local interface_mode=${INTERFACE_LIST[${interface}]}

    #clear old profile
    if [ -n "${T_PROFILE}" ]; then
        cfg_unset "${T_PROFILE}"
    fi

    declare -A dialog_window_conf

    basic_dialog_cmd="dialog --title '${title}' --clear \
            --backtitle '$BACKTITLE' \
            --cancel-label 'Exit' \
            --help-button \
            --colors \
            --menu '\nSpace/Enter to confirm, ESC to go to the previous page' 30 100 30 \
            2>&1 > /dev/tty"

    ###### Change to [Interface] section
    local trasform_profile_name=
    trasform_profile_name=$(cfg_transform_key "${profile}")

    cfg_section_"${trasform_profile_name}"
    T_PROFILE=${trasform_profile_name}

    while true; do
        dialog_cmd=${basic_dialog_cmd}
        index=1

        key="ip-method"
        local ip_method=
        ip_method=$(ini_get_value ${key})

        key="ipv4-method"
        local ipv4_method=
        ipv4_method=$(ini_get_value ${key})

        local dhcp_server=${DHCP_SERVER}

        if [ "${INTERFACE_DEVICE_TYPE}" = "Modem" ]; then
            local signal_thr_type_key="signal.monitor-threshold-type"
            SIGNAL_MONITOR_THR_TYPE=$(ini_get_value "${signal_thr_type_key}")

            local signal_thr_key="signal.monitor-threshold"
            SIGNAL_MONITOR_THR=$(ini_get_value "${signal_thr_key}")

            #transfor cell signal therthod format
            if [ "${SIGNAL_MONITOR_THR_TYPE}" = "SIGNAL-STRENGTH" ]; then
                if [[ ${SIGNAL_MONITOR_THR} =~ ${INT_REGEX} ]]; then
                    SIGNAL_MONITOR_THR=FAIR
                    ini_set_value "${T_PROFILE}" "${signal_thr_key}" ${SIGNAL_MONITOR_THR}
                fi
            else
                if ! [[ ${SIGNAL_MONITOR_THR} =~ ${INT_REGEX} ]]; then
                    SIGNAL_MONITOR_THR=-128
                    ini_set_value "${T_PROFILE}" "${signal_thr_key}" ${SIGNAL_MONITOR_THR}
                fi
            fi

            dialog_cmd+=" '' 'Configure Modem Setting'"
            setup_level2_window "APN" "APN" dialog_profile_apn
            setup_level2_window "SIM" "SIM Slot" dialog_profile_sim
            setup_level2_window "PIN" "PIN Code" dialog_profile_pin
            setup_level2_window "Username" "Username" dialog_profile_username
            setup_level2_window "Password" "Password" dialog_profile_password
            setup_level2_window "RAT" "RAT" dialog_profile_rat
            dialog_cmd+=" '' ' '"

            setup_level0_window "ip-method" "Configure IP Method" dialog_profile_ip_method

        elif [ "${INTERFACE_DEVICE_TYPE}" = "WiFi" ]; then
            dialog_cmd+=" '' 'Configure WiFi Setting'"
            setup_level2_window "SSID" "SSID" dialog_profile_ssid
            setup_level2_window "Password" "Password" dialog_profile_password
            dialog_cmd+=" '' ' '"

            setup_level0_window "ip-method" "Configure IP Method" dialog_profile_ip_method

            if [ "${ip_method}" = "IPV4" ] || [ "${ip_method}" = "IPV4V6" ]; then
                setup_level1_window "ipv4-method" "IPv4 Method" dialog_profile_ipv4_method
                if [ "${ipv4_method}" = "static" ]; then
                    setup_level2_window "ipv4-address" "IPv4 Address" dialog_profile_ipv4_address
                    setup_level2_window "ipv4-netmask" "IPv4 Subnet Mask" dialog_profile_ipv4_netmask

                    if [ "${interface_mode}" = "WAN" ]; then
                        setup_level2_window "ipv4-gateway" "IPv4 Default Gateway" dialog_profile_ipv4_gateway
                        setup_level2_window "primary-dns" "Primary DNS" dialog_profile_ipv4_primary_dns
                        setup_level2_window "secondary-dns" "Secondary DNS" dialog_profile_ipv4_secondary_dns
                    fi
                fi
            fi
            if [ "${ip_method}" = "IPV6" ] || [ "${ip_method}" = "IPV4V6" ]; then
                setup_level1_window "ipv6-method" "IPv6 Method" dialog_profile_ipv6_method
            fi

        elif [ "${INTERFACE_DEVICE_TYPE}" = "WiFi-P2P" ]; then
            dialog_cmd+=" '' 'Configure WiFi-P2P Setting'"
            setup_level2_window "SSID" "SSID" dialog_profile_ssid
            setup_level2_window "Password" "Password" dialog_profile_password
            dialog_cmd+=" '' ' '"

            setup_level0_window "ip-method" "Configure IP Method" dialog_profile_wifi_p2p_ip_method

            setup_level1_window "ipv4-method" "IPv4 Method" dialog_profile_wifi_p2p_ipv4_method
            setup_level2_window "ipv4-address" "IPv4 Address" dialog_profile_ipv4_address
            setup_level2_window "ipv4-netmask" "IPv4 Subnet Mask" dialog_profile_ipv4_netmask

        elif [ "${INTERFACE_DEVICE_TYPE}" = "Ethernet" ]; then
            if [ "${dhcp_server}" = "true" ] && [ "${profile}" = "Profile-1" ] && [ "${interface_mode}" = "LAN" ]; then
                ini_set_value "${T_PROFILE}" "${key}" "static"
                ipv4_method="static"
            fi

            setup_level0_window "ip-method" "Configure IP Method" dialog_profile_ip_method

            if [ "${ip_method}" = "IPV4" ] || [ "${ip_method}" = "IPV4V6" ]; then
                setup_level1_window "ipv4-method" "IPv4 Method" dialog_profile_ethernet_ipv4_method
                if [ "${ipv4_method}" = "static" ]; then
                    setup_level2_window "ipv4-address" "IPv4 Address" dialog_profile_ipv4_address
                    setup_level2_window "ipv4-netmask" "IPv4 Subnet Mask" dialog_profile_ipv4_netmask

                    if [ "${interface_mode}" = "WAN" ]; then
                        setup_level2_window "ipv4-gateway" "IPv4 Default Gateway" dialog_profile_ipv4_gateway
                        setup_level2_window "primary-dns" "Primary DNS" dialog_profile_ipv4_primary_dns
                        setup_level2_window "secondary-dns" "Secondary DNS" dialog_profile_ipv4_secondary_dns
                    fi
                fi
            fi
            if [ "${ip_method}" = "IPV6" ] || [ "${ip_method}" = "IPV4V6" ]; then
                setup_level1_window "ipv6-method" "IPv6 Method" dialog_profile_ipv6_method
            fi
        fi

        if [ "${interface_mode}" = "WAN" ]; then
            dialog_cmd+=" '' ' '"

            if [ "${INTERFACE_DEVICE_TYPE}" = "Ethernet" ]; then
                setup_level0_window "connection.check-method" "Configure Connection Check-alive Method" dialog_profile_ethernet_check_method
            else
                setup_level0_window "connection.check-method" "Configure Connection Check-alive Method" dialog_profile_check_method
            fi

            key="connection.check-method"
            local connection_check_method=
            connection_check_method=$(ini_get_value ${key})

            if [ "${connection_check_method}" = "ping" ]; then
                setup_level2_window "ping.ipv4-target-host" "IPv4 Ping Target Host" dialog_profile_ipv4_target_host
                setup_level2_window "ping.ipv6-target-host" "IPv6 Ping Target Host" dialog_profile_ipv6_target_host
                setup_level2_window "ping.timeout" "Ping Timeout" dialog_profile_ping_timeout
                setup_level2_window "connection.check-alive-interval" "Configure Check-alive Check Interval" dialog_profile_connection_check_alive_interval
                setup_level2_window "connection.check-failed-threshold" "Configure Check-alive Retry Threshold" dialog_profile_connection_check_failed_threshold
            fi

            if [ "${connection_check_method}" = "ping-signalmonitor" ]; then
                setup_level2_window "ping.ipv4-target-host" "IPv4 Ping Target Host" dialog_profile_ipv4_target_host
                setup_level2_window "ping.ipv6-target-host" "IPv6 Ping Target Host" dialog_profile_ipv6_target_host
                setup_level2_window "ping.timeout" "Ping Timeout" dialog_profile_ping_timeout
                setup_level2_window "connection.check-alive-interval" "Configure Check-alive Check Interval" dialog_profile_connection_check_alive_interval
                setup_level2_window "connection.check-failed-threshold" "Configure Check-alive Retry Threshold" dialog_profile_connection_check_failed_threshold

                if [ "${INTERFACE_DEVICE_TYPE}" = "Modem" ]; then
                    setup_level2_window "signal.monitor-threshold-type" "Configure Signal Monitor Threshold Type" dialog_profile_signal_monitor_cell_threshold_type
                    setup_level2_window "signal.monitor-threshold" "Configure Signal Monitor Threshold" dialog_profile_cell_signal_monitor_threshold
                elif [ "${INTERFACE_DEVICE_TYPE}" = "WiFi" ]; then
                    setup_level2_window "signal.monitor-threshold-type" "Configure Signal Monitor Threshold Type" dialog_profile_signal_monitor_wifi_threshold_type
                    setup_level2_window "signal.monitor-threshold" "Configure Signal Monitor Threshold" dialog_profile_signal_monitor_threshold
                fi

                setup_level2_window "signal.monitor-long-interval" "Configure Signal Monitor Long Interval" dialog_profile_signal_monitor_long_interval
            fi

            if [ "${connection_check_method}" = "check-ip-exist-signalmonitor" ]; then

                if [ "${INTERFACE_DEVICE_TYPE}" = "Modem" ]; then
                    setup_level2_window "signal.monitor-threshold-type" "Configure Signal Monitor Threshold Type" dialog_profile_signal_monitor_cell_threshold_type
                    setup_level2_window "signal.monitor-threshold" "Configure Signal Monitor Threshold" dialog_profile_cell_signal_monitor_threshold
                elif [ "${INTERFACE_DEVICE_TYPE}" = "WiFi" ]; then
                    setup_level2_window "signal.monitor-threshold-type" "Configure Signal Monitor Threshold Type" dialog_profile_signal_monitor_wifi_threshold_type
                    setup_level2_window "signal.monitor-threshold" "Configure Signal Monitor Threshold" dialog_profile_signal_monitor_threshold
                fi
                setup_level2_window "signal.monitor-long-interval" "Configure Signal Monitor Long Interval" dialog_profile_signal_monitor_long_interval
            fi
        fi

        dbg "${dialog_cmd}"

        local next_dialog_window=

        select=$(eval "${dialog_cmd}")
        result=$?

        if [ $result -eq $CANCEL ]; then
            dialog_exit
            exit 0
        elif [ $result -eq $ESC ]; then
            return 0
        elif [ $result -eq $HELP ]; then
            dialog_configure_profile_help
            continue
        elif [ $result -ne $OK ]; then
            echo "No Know Key"
            exit 1
        fi

        dbg "Select:${select}"
        next_dialog_window=${dialog_window_conf[" ${select} "]}
        dbg "Next Window Name:${next_dialog_window}"

        if [ -n "${next_dialog_window}" ]; then
            eval "${next_dialog_window}"
        fi
    done
}

dialog_configure_interface_help() {
    local text=
    text="\Z4Select Network Type\Zn
    \n1.  WAN - When set to WAN, this interface will be added to the default gateway list and allow MCM to apply automatic keep-alive and failover/failback control over it.
    \n\n2.  LAN- When set to LAN, MCM will connect this interface using the network attributes defined in Profile-1 and DHCP server can be enabled for this interface.
    \n\n3.  LAN Bridge- Bridge two or more LAN interfaces to construct a larger LAN.
    \n\n4.  Manual - When set to Manual, it allows the user to have total control over this interface. MCM will connect this interface one-time only network attributes defined in Profile-1.MCM will not set these interfaces as default gateway nor apply connection keep-alive and failover/failback control over it.
    \n\n5.  None - MCM will not manage this interface
    \n\n\Z4Configure Network Profile Priority\Zn
    \nWhen the network profile set as \Z41st Priority\Zn cannot connect or the connection becomes unavailable, MCM will automatically \Z4failover\Zn to the next profile in the priority list.
    \n\n\Z4Configure Profile Retry Threshold\Zn
    \nThis value (in seconds) determines the maximum attempts MCM will try to connect using a network profile before failover to the next profile in the priority list.
    \n\n\Z4Configure Profile Timeout\Zn
    \nThis value (in seconds) determines the maximum time MCM will try to connect using a network profile before determining the connection is unavailable.     
    \n\n\Z4Enable/Disable DHCP Server\Zn
    \nDHCP server for LAN or bridged LAN interfaces."

    help_box "${text}"
}

find_modem_id_by_imei() {
    local target_imei=$1
    local modem_index=
    local search_device=
    local modem_index_list=

    modem_index_list=$(mmcli -L | cut -d '/' -f6 | cut -d ' ' -f 1 | tr '\n' ' ')

    if [ -z "${modem_index_list}" ]; then
        echo "Not found Modem on IMEI" "${target_imei}"
        exit "$ERR_GENERIC_FAIL"
    fi

    for modem_index in ${modem_index_list}; do
        if [[ ! ${modem_index} =~ ${UNIT_REGEX} ]]; then
            echo "No fount Modem on IMEI" "${target_imei}"
            exit "$ERR_GENERIC_FAIL"
        fi

        imei=$(mmcli -m "$modem_index" -K | grep "modem.generic.equipment-identifier")
        if [ -z "${imei}" ]; then
            echo "No fount Modem on IMEI" "${target_imei}"
            exit "$ERR_GENERIC_FAIL"
        fi

        search_device=$(echo "${imei}" | grep "${target_imei}")
        if [ -n "${search_device}" ]; then
            echo "$modem_index"
            return
        fi
    done
    echo "No fount Modem on IMEI" "${target_imei}"
}

dialog_configure_interface() {
    local interface=$1
    local title="Configure ${interface} Interfaces"
    local interface_mode=${INTERFACE_LIST[${interface}]}
    local interface_conf_file=
    local basic_dialog_cmd=
    local dialog_cmd=
    local index=1
    local key=
    local imei=

    DHCP_SERVER=

    if [ -n "${INTERFACE}" ]; then
        temporary_interface_configure_file "${INTERFACE}"
    fi

    INTERFACE=${interface}
    INTERFACE_DEVICE_TYPE=$(mx-connect-mgmt nwk_info "${interface}" | grep "Device Type" | cut -d ':' -f 2 | tr -d ' ')

    interface_conf_file=${INTERFACE_CONF_PATH}/${interface}.conf.bk
    if [ ! -f "${interface_conf_file}" ]; then
        interface_conf_file=${INTERFACE_CONF_PATH}/${interface}.conf
    fi

    local profile_list=
    profile_list=$(grep "\[Profile" "${interface_conf_file}" | tr -d '][')

    # shellcheck disable=SC2206
    INTERFACE_PROFILE_LIST=(${profile_list})

    if [ "${INTERFACE_DEVICE_TYPE}" = "Modem" ]; then
        imei=$(mx-connect-mgmt nwk_info -a "${interface}" | grep "IMEI" | cut -d ':' -f 2 | tr -d ' ')
        modem_id=$(find_modem_id_by_imei "${imei}")
        index=0

        MODEM_SUPPORT_RAT_LIST[${index}]="AUTO"
        index=$((index + 1))

        message=$(mmcli -m "${modem_id}" -K | grep modem.generic.supported-modes.value)
        #2G
        support_rat=$(echo "${message}" | grep "2g")
        if [ -n "${support_rat}" ]; then
            MODEM_SUPPORT_RAT_LIST[${index}]="2G"
            index=$((index + 1))
        fi
        support_rat=$(echo "${message}" | grep "3g")
        if [ -n "${support_rat}" ]; then
            MODEM_SUPPORT_RAT_LIST[${index}]="3G"
            index=$((index + 1))
        fi
        support_rat=$(echo "${message}" | grep "4g")
        if [ -n "${support_rat}" ]; then
            MODEM_SUPPORT_RAT_LIST[${index}]="4G"
            index=$((index + 1))
        fi
        support_rat=$(echo "${message}" | grep "5g")
        if [ -n "${support_rat}" ]; then
            MODEM_SUPPORT_RAT_LIST[${index}]="5G"
            index=$((index + 1))
        fi
    fi

    ###############################
    #Parsing interface configure file
    dbg "Interface Configure file:${interface_conf_file}"
    cfg_unset
    cfg_clear
    cfg_parser "${interface_conf_file}"

    ################################

    declare -A dialog_window_conf

    basic_dialog_cmd="dialog --title '${title}' --clear \
            --backtitle '$BACKTITLE' \
            --cancel-label 'Exit' \
            --help-button \
            --colors \
            --menu '\nSpace to toggle, Enter to confirm, ESC to go to the previous page' 30 100 30 \
            2>&1 > /dev/tty"

    ###### Change to [Interface] section
    cfg_section_Interface

    while true; do
        dialog_cmd=${basic_dialog_cmd}
        index=1

        interface_mode=${INTERFACE_LIST[${interface}]}
        if [ -z "${interface_mode}" ]; then
            interface_mode="None"
        fi

        ######
        dialog_cmd+=" ' ${index}' 'Select Network Type for ${interface} [${interface_mode}]'"
        eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_mode_select ${interface_mode}'"
        index=$((index + 1))

        if [ "${INTERFACE_DEVICE_TYPE}" == "Modem" ]; then
            key="gps.default.enabled"
            local gps_default_val=
            local gps_default_status="Disabled"
            local gps_default_default="Disable"

            gps_default_val=$(ini_get_value ${key})
            if [ "${gps_default_val}" = "true" ]; then
                gps_default_status="Enabled"
                gps_default_default="Enable"
            fi

            dialog_cmd+=" ' ${index}' 'Determine the default status of GPS [${gps_default_status}]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_gps_default_enabled ${gps_default_default}'"
            index=$((index + 1))

            key="gps.antenna.type"
            local gps_antenna_default_val=
            gps_antenna_default_val=$(ini_get_value ${key})

            dialog_cmd+=" ' ${index}' 'Determine the default antenna type of GPS [${gps_antenna_default_val}]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_gps_antenna_type ${gps_antenna_default_val}'"
            index=$((index + 1))
        fi

        if [ "${interface_mode}" = "None" ]; then
            echo ""
        else
            if [ "${interface_mode}" = "WAN" ]; then
                ######
                #key="connection.always-keep-alive"
                #local keep_alive=$(ini_get_value ${key})
                #local keep_alive_enabled="Disabled"
                #local keep_alive_default="Disable"
                #if [ ${keep_alive} = "true" ]; then
                #    keep_alive_enabled="Enabled"
                #    keep_alive_default="Enable"
                #fi

                #dialog_cmd+=" ' ${index}' 'Enable/Disable Connection Keep-alive [${keep_alive_enabled}]'"
                #eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_keepalive_enabled ${keep_alive_default}'"
                #index=$((index + 1))

                ######
                key="profile.failover-priority"
                local profiles_org=
                local profiles=
                local profiles_string=
                local count=0

                profiles_org=$(ini_get_value ${key})
                profiles="${profiles_org//,/\ }"

                for i in ${profiles}; do
                    if [ $count -ne 0 ]; then
                        profiles_string+=" ,"
                    fi
                    profiles_string+=Profile-${i}
                    count=$((count + 1))
                done
                dialog_cmd+=" ' ${index}' 'Configure Network Profile Priority [${profiles_string}]'"
                eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_profile_priority \"${profiles_org}\"'"
                index=$((index + 1))

                ###### Profile Retry Threshold
                setup_level0_window "profile.retry-threshold" "Configure Network Profile Retry Threshold" dialog_interface_retry_threshold
            fi

            ####### Prifle Wait Connection Timeout
            key="profile.wait-connected-timeout"
            local wait_connected_timeout=
            wait_connected_timeout=$(ini_get_value ${key})
            dialog_cmd+=" ' ${index}' 'Configure Network Profile Timeout [${wait_connected_timeout}] Seconds'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_profile_timeout \"${wait_connected_timeout}\"'"
            index=$((index + 1))

            if [ "${interface_mode}" = "LAN" ]; then
                ######## DHCP Server
                key="dhcp-server.enabled"
                local dhcp_server=
                dhcp_server=$(ini_get_value ${key})
                DHCP_SERVER=${dhcp_server}
                if [ "${dhcp_server}" = "true" ]; then
                    dialog_cmd+=" ' ${index}' 'Enable/Disable DHCP Server [Enabled]'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_dhcp_server_enabled Enable '"
                    index=$((index + 1))

                    if [ "${INTERFACE_DEVICE_TYPE}" != "WiFi-P2P" ]; then
                        setup_level1_window "dhcp-server.ip-lease-range" "DHCP Server IP Range" dialog_dhcp_ip_range
                        setup_level1_window "dhcp-server.ip-lease-time" "DHCP Server IP Lease Time" dialog_interface_dhcp_lease_time
                    fi
                else
                    dialog_cmd+=" ' ${index}' 'Enable/Disable DHCP Server [Disabled]'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_dhcp_server_enabled Disable '"
                    index=$((index + 1))
                fi

            elif [ "${interface_mode}" = "LAN-Bridge" ]; then
                local ipv4_address=${BRIDGE_IPV4_ADDRESS}
                dialog_cmd+=" ' ${index}' 'Bridge IPv4 Address [${ipv4_address}]'"
                eval "dialog_window_conf[ ' ${index}' ]='dialog_bridge_ipv4_address \"${ipv4_address}\"'"
                index=$((index + 1))

                local ipv4_netmask=${BRIDGE_IPV4_NETMASK}
                dialog_cmd+=" ' ${index}' 'Bridge  IPv4 Subnet Mask [${ipv4_netmask}]'"
                eval "dialog_window_conf[ ' ${index}' ]='dialog_bridge_ipv4_netmask \"${ipv4_netmask}\"'"
                index=$((index + 1))

                ######## Bridge DHCP Server
                local dhcp_server=${BRIDGE_DHCP_ENABLED}
                if [ "${dhcp_server}" = "true" ]; then
                    dialog_cmd+=" ' ${index}' 'Enable/Disable DHCP Server [Enabled]'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_bridge_dhcp_server_enabled Enable '"
                    index=$((index + 1))

                    local dhcp_range=${BRIDGE_IP_RANGE}
                    dialog_cmd+=" ' ${index}' '--- DHCP Server IP Range [${dhcp_range}]'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_bridge_dhcp_ip_range \"${dhcp_range}\" '"
                    index=$((index + 1))

                    local dhcp_lease_time=${BRIDGE_IP_LEASE_TIME}
                    dialog_cmd+=" ' ${index}' '--- DHCP Server IP Lease Time [${dhcp_lease_time}]'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_bridge_dhcp_lease_time \"${dhcp_lease_time}\" '"
                    index=$((index + 1))

                else
                    dialog_cmd+=" ' ${index}' 'Enable/Disable DHCP Server [Disabled]'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_bridge_dhcp_server_enabled Disable '"
                    index=$((index + 1))
                fi
            fi

            if [ "${interface_mode}" != "LAN-Bridge" ]; then
                dialog_cmd+=" '' ' '"
                dialog_cmd+=" ''  '\Z4Network Profile :'"

                if [ "${interface_mode}" = "WAN" ]; then
                    dialog_cmd+=" ' ${index}' '--- Add a New Profile'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_add_profile'"
                    index=$((index + 1))
                    dialog_cmd+=" ' ${index}' '--- Remove a Profile'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_interface_remove_profile'"
                    index=$((index + 1))

                    local profile_list=
                    profile_list=${INTERFACE_PROFILE_LIST[*]}
                    for profile in ${profile_list}; do
                        dialog_cmd+=" ' ${index}' '--- ${profile}'"
                        eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_profile \"${profile}\"'"
                        index=$((index + 1))
                    done
                else
                    dialog_cmd+=" ' ${index}' '--- Profile-1'"
                    eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_profile \"Profile-1\"'"
                    index=$((index + 1))
                fi
            fi
        fi

        dbg "${dialog_cmd}"

        local next_dialog_window=

        select=$(eval "${dialog_cmd}")
        result=$?

        if [ $result -eq $CANCEL ]; then
            dialog_exit
            exit 0
        elif [ $result -eq $ESC ]; then
            return 0
        elif [ $result -eq $HELP ]; then
            dialog_configure_interface_help
            continue
        elif [ $result -ne $OK ]; then
            echo "No Know Key"
            exit 1
        fi

        dbg "Select:${select}"
        next_dialog_window=${dialog_window_conf[" ${select} "]}
        dbg "Next Window Name:${next_dialog_window}"

        if [ -n "${next_dialog_window}" ]; then
            eval "${next_dialog_window}"
        fi
    done
}

dialog_configure_wan_priority() {
    local priority_id=$1
    local title="Configure WAN Interface Failover Priority"
    local default=$2
    #local configure_data=(${3})
    local select=
    local dialog_cmd=
    local index=0

    IFS=" " read -r -a configure_data <<<"$3"

    configure_data+=("--REMOVE--")
    dialog_cmd="dialog --title '${title}' --clear \
        --backtitle '$BACKTITLE' \
        --cancel-label 'Exit' \
        --radiolist 'Select ${priority_id} Priority Interface:\n(Space to toggle, Enter to confirm, ESC to go to the previous page)' 15 70 10 \
        2>&1 > /dev/tty"

    for conf in "${configure_data[@]}"; do
        if [ "${default}" = "${conf}" ]; then
            dialog_cmd+=" ${index} ${conf} on "
        else
            dialog_cmd+=" ${index} ${conf} off "
        fi
        index=$((index + 1))
    done

    dbg "${dialog_cmd}"

    select=$(eval "${dialog_cmd}")
    result=$?

    if [ $result -eq $CANCEL ]; then
        dialog_exit
        exit 0
    elif [ $result -eq $ESC ]; then
        return 0
    elif [ $result -ne $OK ]; then
        echo "No Know Key"
        exit 1
    fi

    dbg "Select:${select}"
    interface=${configure_data[${select}]}
    if [ "${interface}" = "--REMOVE--" ]; then
        case $priority_id in
        1st)
            FAILOVER_LIST[0]=""
            ;;
        2nd)
            FAILOVER_LIST[1]=""
            ;;
        3rd)
            FAILOVER_LIST[2]=""
            ;;
        4th)
            FAILOVER_LIST[3]=""
            ;;
        esac
        INTERFACE_LIST[${default}]="None"
    else
        if [ -n "${default}" ]; then
            INTERFACE_LIST[${default}]="None"
        fi
        if [ "${INTERFACE_LIST[${interface}]}" != "WAN" ]; then
            INTERFACE_LIST[${interface}]="WAN"
        fi

        case $priority_id in
        1st)
            FAILOVER_LIST[0]=${interface}

            for i in 1 2 3; do
                if [ "${FAILOVER_LIST[$i]}" = "${interface}" ]; then
                    FAILOVER_LIST[$i]=""
                fi
            done
            ;;
        2nd)
            FAILOVER_LIST[1]=${interface}
            for i in 0 2 3; do
                if [ "${FAILOVER_LIST[$i]}" = "${interface}" ]; then
                    FAILOVER_LIST[$i]=""
                fi
            done
            ;;
        3rd)
            FAILOVER_LIST[2]=${interface}
            for i in 0 1 3; do
                if [ "${FAILOVER_LIST[$i]}" = "${interface}" ]; then
                    FAILOVER_LIST[$i]=""
                fi
            done
            ;;
        4th)
            FAILOVER_LIST[3]=${interface}
            for i in 0 1 2; do
                if [ "${FAILOVER_LIST[$i]}" = "${interface}" ]; then
                    FAILOVER_LIST[$i]=""
                fi
            done
            ;;
        esac
    fi
    local seamless_failover_interfaces_count=0
    for i in 0 1 2 3; do
        if [ -n "${FAILOVER_LIST[$i]}" ]; then
            seamless_failover_interfaces_count=$((seamless_failover_interfaces_count + 1))
        fi
    done

    if [ ${seamless_failover_interfaces_count} -le 1 ]; then
        FAILOVER_SEAMLESS_ENABLED=false
        _disable_all_wan_keepalive
    fi
}

_failback_check_interval_handler() {
    local select=$1
    FAILBACK_CHECK_INTERVAL=${select}
}

dialog_failback_check_interval() {
    local title="Configure Failback Check Interval"
    local default=$1
    local text=
    local regex=${UINT_REGEX}

    text="This value determines how long (in seconds) the higher priority connection should maintain stability before MCM trigger the failback. The purpose is to avoid unstable connections causing frequent failover and failback.
          \n\nEnter to confirm, ESC to go to the previous page"

    _inputbox_check "${title}" "${text}" "${default}" "${regex}" "_failback_check_interval_handler"
}

dialog_network_interfaces_help() {
    local text=
    text="\Z4\ZbConfigure WAN Interface Failover Priority\Zn
          \nMCM will use the WAN interface set to 1st Priority as the default gateway. When the 1st priority interface becomes unavailable, MCM will automatically failover to the next priority interface. \
         \n\n\Z4\ZbSeamless Failover\Zn \
         \nWhen disabled, in the event that primary interface (e.g., WiFi) connection becomes unavailable, MCM will attempt to establish a connection using all the pre-configured profiles for the primary interface before switching to the backup interface (e.g., LTE). Since each profile requires time to establish a connection, and the backup interface also needs time to initialize, there will be a period of downtime during the failback process. \n
         \nWhen enabled, the backup interface (e.g., LTE) will automatically establish a connection and kept alive. When the primary interface (e.g., WiFi) becomes unavailable, MCM will seamlessly switch the default gateway to the already connected backup interface, avoiding any downtime during the failback process. However, it is essential to be aware that using ping to an external server as the keep-alive method for the backup interface might incur some data costs. \
 	 \n\n\Z4\ZbFailback\Zn \
         \nWhen enabled, the backup connection will automatically failback to the highest priority connection when it became available again.
         \n\n\Z4\ZbFailback Check Interval\Zn \
         \nThis value determines how long (in seconds) the higher priority connection should maintain stability before MCM trigger the failback. The purpose is to avoid unstable connections causing frequent failover and failback. \
	 \n\n\Z4\ZbConfiguring Manual DNS Server Address(es)\Zn \
	 \nIf the DHCP servers that the Moxa computer connects to do not provide DNS server information in their configuration, you have the option to manually configure the DNS address(es)"

    help_box "${text}"
}

dialog_network_interfaces() {
    local title="Configure Network Interfaces"
    local basic_dialog_cmd=
    local dialog_cmd=
    local index=1
    local key=

    declare -A dialog_window_conf

    basic_dialog_cmd="dialog --title '${title}' --clear \
            --backtitle '$BACKTITLE' \
            --cancel-label 'Exit' \
            --help-button \
            --colors \
            --menu '\nSpace to toggle, Enter to confirm, ESC to go to the previous page' 40 90 30 \
            2>&1 > /dev/tty"

    while true; do
        dialog_cmd=${basic_dialog_cmd}
        index=1

        dialog_cmd+=" '' '\Z4Configure Network Interfaces :'"

        for interface in "${!INTERFACE_LIST[@]}"; do
            dialog_cmd+=" ' ${index}' ' ${interface}[${INTERFACE_LIST[${interface}]}]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_interface \"${interface}\"'"
            index=$((index + 1))
        done

        dialog_cmd+=" '' ' '"
        dialog_cmd+=" '' '\Z4Configure WAN Interface Failover Priority :'"

        interface_data=
        for interface in "${!INTERFACE_LIST[@]}"; do
            local device_type
            device_type=$(mx-connect-mgmt nwk_info ${interface} | grep "Device Type" | cut -d ':' -f 2 | tr -d ' ')

            if [ "${device_type}" != "WiFi-P2P" ]; then
                interface_data+="${interface}"
                interface_data+=" "
            fi
        done

        interface=${FAILOVER_LIST[0]}
        dialog_cmd+=" ' ${index}' ' 1st Priority [${interface}]'"
        eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_wan_priority 1st \"${interface}\" \"${interface_data}\"'"
        index=$((index + 1))

        #interface_data=(${interface_data[@]/${interface}})
        interface=${FAILOVER_LIST[1]}
        dialog_cmd+=" ' ${index}' ' 2nd Priority [${interface}]'"
        eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_wan_priority 2nd \"${interface}\" \"${interface_data}\" '"
        index=$((index + 1))

        #interface_data=(${interface_data[@]/${interface}})
        interface=${FAILOVER_LIST[2]}
        dialog_cmd+=" ' ${index}' ' 3rd Priority [${interface}]'"
        eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_wan_priority 3rd \"${interface}\" \"${interface_data}\" '"
        index=$((index + 1))

        #interface_data=(${interface_data[@]/${interface}})
        interface=${FAILOVER_LIST[3]}
        dialog_cmd+=" ' ${index}' ' 4th Priority [${FAILOVER_LIST[3]}]'"
        eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_wan_priority 4th \"${interface}\" \"${interface_data}\"'"
        index=$((index + 1))

        dialog_cmd+=" '' ' '"
        if [ "${FAILOVER_SEAMLESS_ENABLED}" = "true" ]; then
            dialog_cmd+=" ' ${index}' 'Enable/Disable Seamless Failover [Enabled]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_seamless_failover_enabled Enable'"
        else
            dialog_cmd+=" ' ${index}' 'Seamless Failover [Disabled]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_seamless_failover_enabled Disable'"
        fi
        index=$((index + 1))

        dialog_cmd+=" '' ' '"
        if [ "${FAILBACK_ENABLED}" = "true" ]; then
            dialog_cmd+=" ' ${index}' 'Failback [Enabled]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_failback_enabled  Enable'"
            index=$((index + 1))
            dialog_cmd+=" ' ${index}' '--- Failback Check Interval  [${FAILBACK_CHECK_INTERVAL}] seconds'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_failback_check_interval \"${FAILBACK_CHECK_INTERVAL}\"'"
        else
            dialog_cmd+=" ' ${index}' 'Failback [Disabled]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_failback_enabled Disable'"
        fi
        index=$((index + 1))

        dialog_cmd+=" '' ' '"
        if [ "${AUTO_TIMESYNC}" != "Disabled" ]; then
            dialog_cmd+=" ' ${index}' 'Auto Timesync [${AUTO_TIMESYNC}]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_auto_timesync ${AUTO_TIMESYNC}'"
            index=$((index + 1))
            dialog_cmd+=" ' ${index}' '--- Auto Timesync [${AUTO_TIMESYNC_INTERVAL}] seconds'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_auto_timesync_interval \"${AUTO_TIMESYNC_INTERVAL}\"'"

            if [ "${AUTO_TIMESYNC}" = "Cellular" ] || [ "${AUTO_TIMESYNC}" = "GPS" ]; then
                index=$((index + 1))
                dialog_cmd+=" ' ${index}' '--- Auto Timesync Interface [${AUTO_TIMESYNC_INTERFACE}]'"
                eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_auto_timesync_interface \"${AUTO_TIMESYNC_INTERFACE}\"'"
            fi
        else
            dialog_cmd+=" ' ${index}' 'Auto Timesync [Disabled]'"
            eval "dialog_window_conf[ ' ${index}' ]='dialog_configure_auto_timesync Disabled'"
        fi
        index=$((index + 1))

        dialog_cmd+=" '' ' '"
        dialog_cmd+=" ' ${index}' 'Configuring Manual DNS Server Address(es)[${GLOBAL_DNS_SERVERS}]'"
        eval "dialog_window_conf[ ' ${index}' ]='dialog_global_dns_servers \"${GLOBAL_DNS_SERVERS}\"'"
        index=$((index + 1))

        dbg "${dialog_cmd}"
        local next_dialog_window=

        select=$(eval "${dialog_cmd}")
        result=$?

        if [ $result -eq $CANCEL ]; then
            dialog_exit
            exit 0
        elif [ $result -eq $ESC ]; then
            return 0
        elif [ $result -eq $HELP ]; then
            dialog_network_interfaces_help
            continue
        elif [ $result -ne $OK ]; then
            echo "No Know Key"
            exit 1
        fi

        dbg "Select:${select}"
        next_dialog_window=${dialog_window_conf[" ${select} "]}
        dbg "Next Window Name:${next_dialog_window}"

        if [ -n "${next_dialog_window}" ]; then
            eval "${next_dialog_window}"
        fi
    done
}

dialog_main_menu() {
    local title="Moxa Connection Manager (MCM)"
    local basic_dialog_cmd=
    local dialog_cmd=
    local key=

    basic_dialog_cmd="dialog --title '${title}' --clear \
            --backtitle '$BACKTITLE' \
            --cancel-label 'Exit' \
            --menu '\nSelect Operation (Enter to confirm and ESC to exit):' 20 80 20 \
            2>&1 > /dev/tty"

    while true; do

        dialog_cmd=${basic_dialog_cmd}

        dialog_cmd+=" 1 'Configure Network Interfaces'"
        dialog_cmd+=" 2 'Configure Log level [${LOG_LEVEL}]'"
        dialog_cmd+=" 3 'Set to Default MCM Configuration'"
        dbg "${dialog_cmd}"

        select=$(eval "${dialog_cmd}")
        result=$?

        if [ $result -eq $CANCEL ]; then
            dialog_exit
            exit 0
        elif [ $result -eq $ESC ]; then
            dialog_exit
            exit 0
        elif [ $result -ne $OK ]; then
            echo "No Know Key ($result)"
            exit 1
        fi

        dbg "Select:${select}"
        case ${select} in
        1)
            dialog_network_interfaces
            ;;
        2)
            dialog_configure_log_level "${LOG_LEVEL}"
            ;;
        3)
            rm -rf ${CONF_PATH:?}/*
            mx-connect-mgmt default
            parser_main_setting
            parser_interface_setting
            ;;
        esac
    done
}

parser_interface_setting() {
    local interface_list

    interface_list=$(mx-connect-mgmt ls | grep InterfaceName | cut -d ":" -f 2)

    for interface in ${interface_list}; do
        local interface_conf_file
        local always_keep_alive

        interface_conf_file=${INTERFACE_CONF_PATH}/${interface}.conf.bk
        if [ ! -f "${interface_conf_file}" ]; then
            interface_conf_file=${INTERFACE_CONF_PATH}/${interface}.conf
        fi

        always_keep_alive=$(grep ^connection.always-keep-alive "${interface_conf_file}" | cut -d '=' -f 2 | sed 's/,/\ /g')
        if [ "${always_keep_alive}" = "true" ]; then
            INTERFACE_AL_KEEPALIVE_LIST["${interface}"]="true"
        else
            INTERFACE_AL_KEEPALIVE_LIST["${interface}"]="false"
        fi
    done
}

parser_main_setting() {
    local interface_list=
    local main_ini_file=
    local failover_list=
    local onshot_list=
    local bridge_list=
    local standalone_list=

    interface_list=$(mx-connect-mgmt ls | grep InterfaceName | cut -d ":" -f 2)
    main_ini_file=${CONF_PATH}/MoxaConnectionManager.conf
    failover_list=$(grep ^failover.interface.priority ${main_ini_file} | cut -d '=' -f 2 | sed 's/,/\ /g')
    onshot_list=$(grep ^manual.interface.list ${main_ini_file} | cut -d '=' -f 2 | sed 's/,/\ /g')
    bridge_list=$(grep ^bridge.interface.list ${main_ini_file} | cut -d '=' -f 2 | sed 's/,/\ /')
    standalone_list=$(grep ^standalone.interface.list ${main_ini_file} | cut -d '=' -f 2 | sed 's/,/\ /g')

    local index=0
    local key=

    FAILOVER_LIST=()
    INTERFACE_LIST=()

    for interface in ${failover_list}; do
        local check_interface=
        check_interface=$(echo "${interface_list}" | grep "${interface}")

        if [ -n "${check_interface}" ]; then
            INTERFACE_LIST[${interface}]="WAN"
            FAILOVER_LIST[${index}]=${interface}
            index=$((index + 1))
        fi
    done

    for interface in ${onshot_list}; do
        local check_interface=
        check_interface=$(echo "${interface_list}" | grep "${interface}")

        if [ -n "${check_interface}" ]; then
            INTERFACE_LIST["${interface}"]="Manual"
        fi
    done

    for interface in ${bridge_list}; do
        local check_interface=
        check_interface=$(echo "${interface_list}" | grep "${interface}")

        if [ -n "${check_interface}" ]; then
            INTERFACE_LIST["${interface}"]="LAN-Bridge"
        fi
    done

    for interface in ${standalone_list}; do
        local check_interface=
        check_interface=$(echo "${interface_list}" | grep "${interface}")

        if [ -n "${check_interface}" ]; then
            INTERFACE_LIST["${interface}"]="LAN"
        fi
    done

    for interface in ${interface_list}; do
        if [ "${interface}" = "MCMLAN" ]; then #skip bridge interface
            continue
        fi

        if [ -z ${INTERFACE_LIST["${interface}"]} ]; then
            INTERFACE_LIST["${interface}"]="None"
        fi
    done

    ###############################
    #Parsing  Main  configure file
    local main_conf_file=${CONF_PATH}/MoxaConnectionManager.conf
    dbg "Main Configure file:${main_conf_file}"
    cfg_unset
    cfg_clear
    cfg_parser ${main_conf_file}

    ######### change [Main] section
    cfg_section_Main

    key="log-level"
    LOG_LEVEL=$(ini_get_value ${key})

    key="auto-timesync"
    AUTO_TIMESYNC=$(ini_get_value ${key})

    key="auto-timesync.interval"
    AUTO_TIMESYNC_INTERVAL=$(ini_get_value ${key})

    key="auto-timesync.interface"
    AUTO_TIMESYNC_INTERFACE=$(ini_get_value ${key})

    cfg_section_WAN
    key="failback.enabled"
    FAILBACK_ENABLED=$(ini_get_value ${key})

    key="failback.check-interval"
    FAILBACK_CHECK_INTERVAL=$(ini_get_value ${key})

    key="failover.seamless"
    FAILOVER_SEAMLESS_ENABLED=$(ini_get_value ${key})

    key="global.dns.servers"
    GLOBAL_DNS_SERVERS=$(ini_get_value "${key}")

    cfg_section_LAN
    key="dhcp-server.enabled"
    BRIDGE_DHCP_ENABLED=$(ini_get_value ${key})
    key="ipv4-address"
    BRIDGE_IPV4_ADDRESS=$(ini_get_value ${key})
    key="ipv4-netmask"
    BRIDGE_IPV4_NETMASK=$(ini_get_value ${key})
    key="dhcp-server.ip-lease-range"
    BRIDGE_IP_RANGE=$(ini_get_value ${key})
    key="dhcp-server.ip-lease-time"
    BRIDGE_IP_LEASE_TIME=$(ini_get_value ${key})

    cfg_unset
    cfg_clear
}

init() {
    parser_main_setting
    parser_interface_setting
}

main() {
    init
    dialog_main_menu
}

main "$@"
