#!/bin/bash -e
#
# Copyright (C) 2023 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#

set -e

source "${INSTL_SCRIPT_DIR}/lib/util.sh"
source "${INSTL_SCRIPT_DIR}/lib/logger.sh"
source "${INSTL_SCRIPT_DIR}/lib/version.sh"
source "${INSTL_SCRIPT_DIR}/lib/driver.sh"
log::init "${INSTL_LOG_FILE}" "${INSTL_LOG_LEVEL}"

main() {
        log::info "Rename serial device with ttyMX"
        cat <<EOF >/etc/udev/rules.d/99-moxa-x86-serial-naming.rules
KERNEL=="ttyS0"   DRIVERS=="serial"  MODE="0660" SYMLINK+="ttyM0"
KERNEL=="ttyS1"   DRIVERS=="serial"  MODE="0660" SYMLINK+="ttyM1"
EOF
        {
                udevadm control --reload-rules
                udevadm trigger
        } >>"${INSTL_LOG_FILE}"

        # i915 driver need to force probe on Elkhart lake before kernel 5.15
        if drv::is_loaded "i915" && ! util::is_exists "/dev/dri/card0" && [[ $(ver::compare "$LINUX_KERNEL_VER" "5.15") -lt 0 ]]; then
                log::info "Graphics device is not properly supported by the driver. Force probe it."
                {
                        local device_pci_id

                        for bus in "/sys/devices/pci0000:00/0000"*; do
                                # 0x030000 means VGA Display controller
                                if [[ "$(cat "$bus"/class)" == "0x030000" ]]; then
                                        device_pci_id=$(cat "$bus"/device 2>/dev/null)
                                        break
                                fi
                        done

                        if ! util::is_empty_str "$device_pci_id"; then
                                if ! grep -wq "i915.force_probe" "/etc/default/grub"; then
                                        sed -i "s/GRUB_CMDLINE_LINUX=\"[^\"]*/& i915.force_probe=${device_pci_id#"0x"}/" /etc/default/grub
                                        case "${OS_PACKAGE_MANAGER}" in
                                        "apt") update-grub2 ;;
                                        "yum") grub2-mkconfig -o /boot/grub2/grub.cfg ;;
                                        esac
                                fi

                                if [[ "${LINUX_DISTRO_ID}" == "debian" ]]; then
                                        if add-apt-repository "deb http://deb.debian.org/debian/ bullseye main non-free"; then
                                                apt-get update
                                                apt-get install firmware-linux -y
                                        fi
                                fi
                        else
                                log::warn " Can not get graphics device PCI ID"
                        fi

                } >>"${INSTL_LOG_FILE}" 2>&1
        fi

        log::info "Enable touch panel wake up from suspend"
        # 7"            0eef:c002 D-WAV Scientific Co., Ltd eGalaxTouch P80H46 0070 v0070_T0M0 k4.18.203
        # 10.4"         0eef:c002 D-WAV Scientific Co., Ltd eGalaxTouch P80H46 0104 v0104_T0M0 k4.18.203
        # 12.1" (4:3)   222a:0001 ILI Technology Corp. Multi-Touch Screen
        # 12.1" (16:10) 0eef:c002 D-WAV Scientific Co., Ltd eGalaxTouch P80H46 2306 v20001D013 k4.18.204
        # 15"   (4:3)   0eef:c002 D-WAV Scientific Co., Ltd eGalaxTouch P81X84 0150 vM00_M01 k4.22.239
        # 15.6" (16:9)  0eef:c002 D-WAV Scientific Co., Ltd eGalaxTouch P81X60 2306 v26001D003 k4.18.201
        cat <<EOF >/etc/udev/rules.d/panel-suspend-wakeup.rules
ACTION=="add", SUBSYSTEM=="usb", DRIVERS=="usb", ATTRS{idVendor}=="0eef", ATTRS{idProduct}=="c002", ATTR{power/wakeup}="enabled"
ACTION=="add", SUBSYSTEM=="usb", DRIVERS=="usb", ATTRS{idVendor}=="222a", ATTRS{idProduct}=="0001", ATTR{power/wakeup}="enabled"
EOF

        log::info "Install hardware sensor monitor lm-sensors packages and configuration file"
        {
                case "${OS_PACKAGE_MANAGER}" in
                "apt") apt-get install -y "lm-sensors" ;;
                "yum") yum install -y "lm_sensors" ;;
                esac
        } >>"${INSTL_LOG_FILE}"

        cat <<EOF >/lib/modules-load.d/it87.conf
it87
EOF
        if ! util::is_exists "/etc/sensors.d/"; then
                mkdir -p "/etc/sensors.d/"
        fi

        if ! util::is_exists "/etc/sensors.d/it8786.conf"; then
                adc_lsb_coeff="0.9"
                V3_3_Rin="6.19"
                V3_3_Rf="10"
                V5_Rin="10"
                V5_Rf="10"

                cat <<EOF >/etc/sensors.d/it8786.conf
chip "it8786-*"
    ignore in0
    ignore in1

    label in2 "V3.3"
    compute in2 @*$adc_lsb_coeff*(1+($V3_3_Rin/$V3_3_Rf)), @/($adc_lsb_coeff*(1+($V3_3_Rin/$V3_3_Rf)))
    set in2_min 3.3*0.9
    set in2_max 3.3*1.1

    label in3 "V5.0"
    compute in3 @*$adc_lsb_coeff*(1+($V5_Rin/$V5_Rf)), @/($adc_lsb_coeff*(1+($V5_Rin/$V5_Rf)))
    set in3_min 5.0*0.9
    set in3_max 5.0*1.1

    ignore in4
    ignore in5
    ignore in6
    ignore in7
    # VBat
    ignore in8

    ignore fan1
    ignore fan2
    ignore fan3

    label temp1 "SYSTEM"
    label temp2 "SYSTEM2"

    ignore temp3
    ignore intrusion0
EOF

        fi
}

main
